package com.xebialabs.plugin.manager.service

import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.config.PluginManagerProperties
import com.xebialabs.plugin.manager.event.{PluginInstalledEvent, PluginUpdatedEvent}
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata}
import com.xebialabs.plugin.manager.model.DbPlugin
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig, NexusServerConfig}
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.rest.dto.PluginDto
import com.xebialabs.plugin.manager.service.PluginService.defaultTimeout
import com.xebialabs.plugin.manager.startup.PluginReloader
import com.xebialabs.plugin.manager.validator.{PluginValidator, TypeSystemValidator}
import com.xebialabs.plugin.manager.{Plugin, PluginId, PluginManager}
import grizzled.slf4j.Logging
import org.springframework.context.ApplicationEventPublisher

import java.io.File
import javax.annotation.PreDestroy
import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.jdk.CollectionConverters._
import scala.language.postfixOps
import scala.util.{Failure, Success, Try}

trait PluginService extends Logging {
  def typeSystemValidator: TypeSystemValidator

  def pluginManager: PluginManager

  def pluginReloader: PluginReloader

  def eventPublisher: ApplicationEventPublisher

  implicit lazy val ec: ExecutionContext = pluginManager.ec

  def repositories: mutable.Map[String, PluginsRepository]

  def addRepository(repository: PluginsRepository): Boolean

  def deleteRepository(name: String): Boolean


  def update(): Unit = {
    if (repositories.nonEmpty) {
      Await.ready(Future.sequence(repositories.values.map(_.update())), defaultTimeout)
    }
  }

  def search(query: Option[String]): Map[ArtifactId, Option[PluginMetadata]] = {
    logger.debug(s"search($query)")
    Await.result(
      for {
        installed <- pluginManager.search(query)
        available <- Future.sequence(repositories.values.map(_.search(query))).map(_.flatten)
      } yield {
        available.foldLeft(installed.map(_ -> Option.empty[PluginMetadata]).toMap) {
          case (acc, (id, pm1)) =>
            acc.get(id) match {
              case None =>
                acc.updated(id, pm1)
              case Some(_) =>
                pm1.map(pm => acc.updated(id, Some(pm))).getOrElse(acc)
            }
        }
      },
      defaultTimeout
    )
  }

  def searchOfficial(query: String): Map[ArtifactId, Option[PluginMetadata]] = {
    Await.result(
      for {
        available <- Future.sequence(repositories.values.map(_.search(Some(query)))).map(_.flatten)
      } yield {
        available.toMap
      },
      defaultTimeout
    )
  }

  def search(query: String): Map[ArtifactId, Option[PluginMetadata]] = search(Option(query).filter(_.nonEmpty))

  def list(): Map[ArtifactId, Option[PluginMetadata]] = search(None)

  def listInstalled(): Seq[PluginId] =
    pluginManager.listInstalled()

  def installOrUpdate(plugin: Plugin): Unit = {
    logger.info(s"Installing plugin ${plugin.id.id}...")
    pluginManager.installOrUpdate(plugin)
    logger.info(s"Installation of plugin ${plugin.id.id} to the database is complete. " +
      s"System restart required for the plugin to be ready for usage.")
  }

  def install(pluginId: PluginId, bytes: Array[Byte]): InstallationResult = {
    logger.info(s"Installing plugin $pluginId...")
    validateInstall(pluginId, bytes) {
      val plugin = Plugin(pluginId, None, bytes)
      // TODO S-88045
      val result = doWithTypeSystemValidation(plugin) {
        pluginManager.install(plugin)
      }
      logger.info(result.message)
      publishPluginManagerEvent(result)
      result
    }
  }

  private def validateInstall(pluginId: PluginId, bytes: Array[Byte])(successCallback: => InstallationResult): InstallationResult = {
    val installedPlugins = listInstalled()
    val sameVersionAlreadyInstalled = installedPlugins.exists(_.equals(pluginId))
    val pluginName = pluginId.name
    val differentVersionAlreadyInstalled = installedPlugins.exists(_.name.equals(pluginName))

    if (sameVersionAlreadyInstalled) {
      val result = SameVersionInstalledAlready(pluginName)
      logger.info(result.message)
      result
    } else if (differentVersionAlreadyInstalled) {
      val result = DifferentVersionInstalledAlready(pluginName)
      logger.info(result.message)
      result
    } else {
      val valid = new PluginValidator().validate(bytes, pluginId)
      if (valid.success) successCallback else valid
    }
  }

  def update(existingPluginName: String, newVersion: PluginId.LocalFile, bytes: Array[Byte]): InstallationResult = {
    logger.info(s"Updating plugin $existingPluginName to version $newVersion...")
    // TODO: S-90359 this is bad.  We should be doing an exact match unless a wildcard is included.  The startsWith is only here because of S-80216
    //  requiring support for wildcards in the name.  This could fail just with plugins called "foo" and "foos" since it would find both when the existingPluginName is "foo"
    //  should use presence of wildcard in "existingPluginName" to drive whether we use equals or startsWith
    val matchingPlugins = listInstalled().filter(plugin => plugin.pluginId.name.startsWith(existingPluginName) && plugin.pluginId.isInstanceOf[LocalFile])
    if (matchingPlugins.size > 1) {
      val result = MultipleInstalledPluginsFoundForUpdate(existingPluginName)
      logger.info(result.message)
      result
    } else if (matchingPlugins.isEmpty) {
      val result = ZeroInstalledPluginsFoundForUpdate(existingPluginName)
      logger.info(result.message)
      result
    } else {
      val alreadyInstalled = matchingPlugins.head.asInstanceOf[LocalFile]
      val newPlugin = Plugin(newVersion, None, bytes)
      val result = doWithTypeSystemValidation(newPlugin) {
        pluginManager.update(alreadyInstalled, newPlugin)
      }
      logger.info(result.message)
      publishPluginManagerEvent(result)
      result
    }
  }

  def installOrUpdateFromRepository(id: PluginId.Artifact): Try[Unit] =
    repositories.get(id.repository).map { repo =>
      Try {
        val plugin = Await.result(repo.get(id), defaultTimeout)
        val installationResult = doWithTypeSystemValidation(plugin) {
          pluginManager.install(plugin)
        }
        publishPluginManagerEvent(installationResult)
        if (!installationResult.success) {
          throw new IllegalStateException(installationResult.message)
        }
      }.recoverWith {
        case err =>
          logger.warn(err.getMessage)
          Failure(err)
      }
    }.getOrElse {
      Failure(new NotFoundException(s"Unknown plugin repository '${id.repository}"))
    }

  private def doWithTypeSystemValidation(plugin: Plugin)(pluginOperation: => Unit): InstallationResult = {
    val validations = typeSystemValidator.validate(plugin)
    // TODO S-88045 - log validations
    if (validations.isFailure) {
      InstallationFailure(plugin.id.name, validations.failureReason)
    } else {
      // requires a restart - i.e. it's a normal InstallationResult
      val result = Try(pluginOperation) match {
        case Failure(exception) =>
          logger.warn(s"Unable install/update ${plugin.id.name}", exception)
          InstallationFailure(plugin.id.name, exception.getMessage)
        case Success(_) =>
          // TODO S-88045 restart needs some message
          InstallationSuccess(plugin.id.name, requiresRestart = true)
      }
      logger.info(result.message)
      if (validations.requiresRestart) {
        logger.warn(s"Restart is needed: ${validations.restartReason}")
        result
      } else {
        // broadcast ReloadPlugins message which is going to invoke pluginReloader.reloadPlugin on all of the nodes
        pluginReloader.reloadPlugin(plugin)
        InstallationSuccess(plugin.id.name, requiresRestart = false)
      }
    }
  }


  def installFromRepository(id: PluginId.Artifact): InstallationResult = {
    try {
      repositories.get(id.repository).map { repo =>
        val plugin = Await.result(repo.get(id), defaultTimeout)
        validateInstall(plugin.id, plugin.bytes) {
          // TODO S-88045 validate changes - if it can be installed immediately - just apply it
          val result = doWithTypeSystemValidation(plugin) {
            pluginManager.install(plugin)
          }
          publishPluginManagerEvent(result)
          result
        }
      }.getOrElse {
        val result = RepositoryNotFound(id.repository, id.name)
        logger.info(result.message)
        result
      }
    } catch {
      case err: Exception =>
        logger.warn(err.getMessage)
        val result = PluginNotFound(id.name)
        logger.info(result.message)
        result
    }
  }

  def updateFromRepository(id: PluginId.Artifact): InstallationResult = {
    def doUpdate(repo: PluginsRepository, newVersion: PluginId.Artifact, existing: PluginId): InstallationResult = {
      val newPlugin = Await.result(repo.get(newVersion), defaultTimeout)
      doWithTypeSystemValidation(newPlugin) {
        pluginManager.update(existing, newPlugin)
      }
    }

    repositories.get(id.repository).map { repo =>
      val installedPlugins = getPluginsToUpdate(id.name)
      val availablePlugins = getUpdateMatchingPlugins(id)

      val updateResult = UpdateStrategy.doUpdate(repo, installedPlugins, availablePlugins, id.name, doUpdate)
      logger.info(updateResult.message)
      publishPluginManagerEvent(updateResult)
      updateResult
    }.getOrElse {
      val result = RepositoryNotFound(id.repository, id.name)
      logger.info(result.message)
      result
    }
  }

  def getPluginsToUpdate(pluginName: String): Seq[PluginId] = {
    val pluginNameCriteria = if (pluginName.endsWith("*")) pluginName.dropRight(1) else pluginName
    listInstalled().filter(plugin => plugin.pluginId.name.startsWith(pluginNameCriteria) && plugin.pluginId.isInstanceOf[Artifact])
  }

  def getUpdateMatchingPlugins(id: PluginId.Artifact): Map[ArtifactId, Option[PluginMetadata]] = {
    if (id.name.endsWith("*")) {
      searchOfficial(id.name.dropRight(1))
    } else {
      searchOfficial(id.name)
        .filter(item => item._1.repository.get.equals(id.repository) && item._1.groupId.equals(id.groupId) && item._1.artifactId.equals(id.artifactId))
    }
  }

  def uninstall(repositoryId: PluginSource.Value,
                groupId: String,
                artifactId: String,
                version: Option[String]): Try[Unit] = {
    logger.info(s"Uninstalling plugin $repositoryId:$groupId:$artifactId-$version")

    // TODO S-88045 how to check if it is ok to uninstall a plugin that is already installed
    pluginManager.getPluginBy(repositoryId, groupId, artifactId, version) match {
      case Some(plugin) =>
        if (pluginManager.uninstall(plugin)) {
          pluginManager.revertPreviousVersionIfNecessary(plugin)
          Success((): Unit)
        } else {
          Failure(PluginUninstallException(plugin, s"Only plugins in ${PluginStatus.READY_FOR_INSTALL} status can be uninstalled."))
        }
      case None =>
        // TODO S-88045 if plugin source is filesystem and we install - then plugin never ends up on fs
        Success((): Unit)
        // Failure(PluginNotFoundException(s"Plugin $repositoryId:$groupId:$artifactId-$version not found in the system."))
    }
  }

  def getLogo(repositoryId: String, artifactId: String): Option[File] =
    for {
      repo <- repositories.get(repositoryId)
      logo <- repo.getLogo(artifactId)
    } yield logo

  def attachMetadata(pluginId: PluginId): PluginDto = {
    val repoId = pluginId.source.toString
    val metadata = for {
      repo <- repositories.get(repoId)
      pluginsMeta <- repo.getMetadata(pluginId.toArtifactId)
    } yield pluginsMeta
    PluginDto(pluginId, metadata)
  }

  def attachStatus: PluginDto => PluginDto = {
    val statuses: Map[String, PluginStatus.Value] = pluginManager.fetchPluginStatuses()
    pluginDto => pluginDto.copy(status = statuses.get(pluginDto.plugin.name))
  }

  def extend(data: Seq[PluginId]): Seq[PluginDto] = data.map(attachMetadata).map(attachStatus)

  @PreDestroy
  def shutdown(): Unit = {
    logger.info("Shutting down LocalPluginManager...")
    Await.ready(
      Future.sequence(repositories.values.collect {
        case nexus: NexusPluginRepository => nexus.shutdown()
      }),
      repositories.size * defaultTimeout
    )
  }

  private def publishPluginManagerEvent(result: InstallationResult): Unit = {
    result match {
      case InstallationSuccess(pluginName, requiresRestart) => eventPublisher.publishEvent(PluginInstalledEvent(this, pluginName, requiresRestart))
      case UpdateSuccess(pluginName, requiresRestart) => eventPublisher.publishEvent(PluginUpdatedEvent(this, pluginName, requiresRestart))
      case _ => ()
    }
  }

}

object PluginService {
  // TODO: get this from configuration
  val defaultTimeout: Duration = 180 seconds

  def serversFromConfig(pluginsConfig: PluginManagerProperties): Map[String, Try[NexusServerConfig]] =
    Option(pluginsConfig.getServers) match {
      case Some(servers) => servers.asScala.collect(server =>
        server.getName -> NexusServerConfig.fromConfig(server)).toMap
      case None => Map.empty
    }

  def repositoriesFromConfig(pluginsConfig: PluginManagerProperties, servers: Map[String, Try[NexusServerConfig]])
                            (implicit productConfig: ProductConfig): Map[String, Try[NexusRepositoryConfig]] =
    Option(pluginsConfig.getRepositories) match {
      case Some(repositories) => repositories.asScala.map(repo =>
        repo.getName -> NexusRepositoryConfig.fromConfig(repo.getName)(repo, servers)).toMap
      case None => Map.empty
    }

  def configuredRepositories(pluginsConfig: PluginManagerProperties)
                            (implicit productConfig: ProductConfig): List[Try[NexusPluginRepository]] =
    repositoriesFromConfig(pluginsConfig, serversFromConfig(pluginsConfig))
      .map { case (name, tryConfig) =>
        tryConfig.map(config => NexusPluginRepository.memCached(name, config))
      }.toList
}


@HttpResponseCodeResult(statusCode = 501)
case class PluginUninstallException(plugin: DbPlugin, message: String) extends DeployitException(s"Unable to uninstall plugin $plugin. $message")

@HttpResponseCodeResult(statusCode = 404)
case class PluginNotFoundException(message: String) extends DeployitException(message)

@HttpResponseCodeResult(statusCode = 400)
case class JARNotFoundException(message: String) extends DeployitException(message)
