package com.xebialabs.plugin.manager.validator

import com.xebialabs.deployit.plugin.api.reflect.IDescriptorRegistry
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.xlplatform.synthetic.TypeDefinitionDocuments._
import com.xebialabs.xlplatform.synthetic.TypeSpecification
import com.xebialabs.xlplatform.synthetic.xml.SyntheticXmlDocument
import com.xebialabs.xlplatform.synthetic.yaml.TypeDefinitionYamlDocument
import org.apache.commons.io.file.PathUtils

import java.io.File
import java.nio.charset.StandardCharsets
import java.util.jar.JarFile
import java.util.zip.{ZipEntry, ZipFile}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try, Using}

abstract class SyntheticExtractor() {
  def extractFrom(file: File, pluginId: PluginId): Option[Try[List[TypeSpecification]]]

  // TODO this is weird as it will make a lookup and define type for the registry
  def descriptorRegistry: IDescriptorRegistry

  def readSynthetic(file: File): Option[Try[List[TypeSpecification]]] = {
    Try(new JarFile(file)).toOption.flatMap(jarFile => {
      val syntheticXml = Option(jarFile.getEntry(SYNTHETIC_XML))
        .map(entry => new String(jarFile.getInputStream(entry).readAllBytes(), StandardCharsets.UTF_8))
        .map(syntheticString => Try(SyntheticXmlDocument.read(syntheticString).getTypes.asScala.toList))

      jarFile.close()
      syntheticXml
    })
  }
}

class JarSyntheticExtractor(val descriptorRegistry: IDescriptorRegistry) extends SyntheticExtractor {
  override def extractFrom(file: File, pluginId: PluginId): Option[Try[List[TypeSpecification]]] = {
    readSynthetic(file)
  }
}

class ReloadableSyntheticExtractor(val descriptorRegistry: IDescriptorRegistry) extends SyntheticExtractor {
  // scalastyle:off cyclomatic.complexity
  override def extractFrom(file: File, pluginId: PluginId): Option[Try[List[TypeSpecification]]] = {
    val result = Using(new ZipFile(file)) { zipFile =>
      val typesXml = Option(zipFile.getEntry(TYPE_DEFINITIONS_XML))
      val typesYaml = Option(zipFile.getEntry(TYPE_DEFINITIONS_YAML))

      if (typesXml.isEmpty && typesYaml.isEmpty) {
        Failure(new IllegalArgumentException(s"Plugin ${pluginId.toString} doesn't have any type-definitions file"))
      } else if (typesXml.isDefined && typesYaml.isDefined) {
        Failure(new IllegalArgumentException(s"Plugin ${pluginId.toString} contains both $TYPE_DEFINITIONS_XML and $TYPE_DEFINITIONS_YAML files"))
      } else {
        val xmlTypeSpecifications = typesXml match {
          case Some(xml) => loadZipEntryContents(zipFile, xml) match {
            case Success(contents) => readXmlDefinition(contents)
            case Failure(ex) => Failure(ex)
          }
          case None => Success(List.empty[TypeSpecification])
        }

        val yamlTypeSpecifications = typesYaml match {
          case Some(yaml) => loadZipEntryContents(zipFile, yaml) match {
            case Success(contents) => readYamlDefinition(contents)
            case Failure(ex) => Failure(ex)
          }
          case None => Success(List.empty[TypeSpecification])
        }

        for {
          xmlTypes <- xmlTypeSpecifications
          yamlTypes <- yamlTypeSpecifications
        } yield xmlTypes ++ yamlTypes
      }
    }

    // Ensure None is never returned because that is considered a valid plugin
    result match {
      case Success(value) => Some(value)
      case Failure(ex) => Some(Failure(ex))
    }
  }
  // scalastyle:on cyclomatic.complexity

  private def loadZipEntryContents(file: ZipFile, entry: ZipEntry): Try[String] = {
    Using(file.getInputStream(entry)) { stream =>
      new String(stream.readAllBytes(), StandardCharsets.UTF_8)
    }
  }

  private def readXmlDefinition(contents: String): Try[List[TypeSpecification]] = {
    Try(SyntheticXmlDocument.read(contents).getTypes.asScala.toList)
  }

  private def readYamlDefinition(contents: String): Try[List[TypeSpecification]] = {
    Try(TypeDefinitionYamlDocument.read(contents).getTypes.asScala.toList)
  }
}

class XldpSyntheticExtractor(val descriptorRegistry: IDescriptorRegistry) extends SyntheticExtractor {

  override def extractFrom(file: File, pluginId: PluginId): Option[Try[List[TypeSpecification]]] = {
    val pluginVersion = if (pluginId.version.isDefined) "-" + pluginId.version.get.id else ""
    val jarName = pluginId.name + pluginVersion + ".jar"

    val xldpFile = new ZipFile(file)
    val jarEntry = getJarEntryFrom(xldpFile, jarName)

    val result = if (jarEntry.isDefined) {
      val jarInputStream = xldpFile.getInputStream(jarEntry.get)
      val pluginFile = ValidatorUtils.createFileInWorkFolderFrom(jarInputStream.readAllBytes(), pluginId.name)
      jarInputStream.close()

      val synthetic = readSynthetic(pluginFile)

      PathUtils.deleteDirectory(pluginFile.getParentFile.toPath, PathUtils.EMPTY_LINK_OPTION_ARRAY)
      synthetic
    } else {
      Some(Failure(new IllegalArgumentException("JAR file doesn't exist : " + jarName)))
    }

    xldpFile.close()
    result
  }

  def getJarEntryFrom(xldpFile: ZipFile, jarName: String): Option[ZipEntry] = {
    // when creating xldp on mac the jar inside can only be extracted by using its name without "./". For official
    // plugins (ziped by gradle) it's the other way around. I have no idea why, so hence this workaround.
    val jarNameStartingWithDotSlash = "./" + jarName
    val jarEntry = Option(xldpFile.getEntry(jarName))

    if (jarEntry.isDefined) jarEntry else Option(xldpFile.getEntry(jarNameStartingWithDotSlash))
  }
}
