package com.xebialabs.plugin.manager.repository.nexus

import java.io.ByteArrayInputStream
import java.nio.charset.StandardCharsets.UTF_8
import java.nio.file.{Path, Paths}
import akka.actor.ActorSystem
import akka.http.scaladsl.model.HttpCharsets.`UTF-8`
import akka.http.scaladsl.model.HttpMethods._
import akka.http.scaladsl.model.MediaTypes.{`application/json`, `application/octet-stream`, `application/zip`}
import akka.http.scaladsl.model._
import akka.http.scaladsl.model.headers.{Accept, Authorization, `Accept-Charset`}
import akka.stream.{Materializer, SystemMaterializer}
import com.typesafe.config.Config
import com.typesafe.config.ConfigFactory
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.metadata._
import com.xebialabs.plugin.manager.repository.CachedPluginRepository
import com.xebialabs.plugin.manager.repository.nexus.NexusPluginRepository.{latestVersion, verifyPlugin}
import com.xebialabs.plugin.manager.repository.nexus.NexusPluginRepositoryException._
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.ListMethod
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.ListMethod.ByGroupId
import com.xebialabs.plugin.manager.repository.storage.{PluginMetadataMemoryStorage, PluginMetadataStorage}
import grizzled.slf4j.Logging
import org.apache.commons.codec.digest.DigestUtils
import spray.json._

import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success, Try}

case class NexusPluginRepository(name: String,
                                 config: NexusRepositoryConfig,
                                 cache: PluginMetadataStorage)
  extends CachedPluginRepository[ArtifactResult](_.extensions.values.headOption.map(_.toPluginId(name)))
    with NexusJsonProtocol
    with MetadataEntry.Protocol
    with Logging {

  val customConfig: Config = ConfigFactory.parseString("akka.coordinated-shutdown.run-by-actor-system-terminate=off")
  val fullConfig: Config = ConfigFactory.load(customConfig)
  implicit val system: ActorSystem = ActorSystem(s"xl-plugin-manager_$name", fullConfig)
  implicit val materializer: Materializer = SystemMaterializer.get(system).materializer
  implicit val ec: ExecutionContext = system.dispatcher

  override def getFromRemote(pluginId: PluginId.Artifact): Future[ByteArrayInputStream] = {
    logger.info(s"$name: Downloading plugin ${pluginId.id}")
    for {
      getSha1 <- downloadSha1(pluginId)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(pluginId, status, msg))

      getPlugin <- downloadArtifact(pluginId)
      pluginContent <- getPlugin
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(pluginId, status, msg))

      (sha1, length) <- verifyPlugin(pluginId, sha1Content, pluginContent, getPlugin.entity.contentLengthOption)
    } yield {
      logger.info(s"$name: Downloaded plugin from ${config.server.uri}: ${pluginId.id} ($length bytes) | SHA1: $sha1")
      new ByteArrayInputStream(pluginContent)
    }
  }

  override def searchFromRemote(query: Option[String]): Future[Seq[ArtifactResult]] = {
    logger.info(s"$name: Searching for '${query.getOrElse("")}'")
    for {
      resp <- searchArtifact(query)
      content <- resp
        .onSuccess(_.asJson[NexusArtifactResults])
        .onFailure((status, msg) => QueryFailed(query, status, msg))
      plugins <- if (content.tooManyResults) {
        Future.failed(TooManyResults(query, content.totalCount))
      } else {
        Future.successful(
          content.data.filter(_.extensions.nonEmpty)
        )
      }
    } yield {
      plugins.foreach { r =>
        logger.trace(s"$name: found ${r.groupId}/${r.artifactId}-${r.version.id} ${r.extensions.keySet.mkString("(", ",", ")")}")
      }
      logger.info(s"$name: Found ${plugins.size} plugins (query: '${query.getOrElse("")}')")
      plugins
    }
  }

  override def fetchMetadata: Future[Map[ArtifactId, ExtendedMetadata]] = {
    logger.info(s"Downloading metadata for plugin repository $name from ${config.server.uri}")
    for {
      jsonId <- searchLatestMetadataArtifact("json", None)

      getSha1 <- downloadSha1(jsonId)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(jsonId, status, msg))

      resp <- downloadArtifact(jsonId)
      content <- resp
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(jsonId, status, msg))

      (sha1, length) <- verifyPlugin(jsonId, sha1Content, content, resp.entity.contentLengthOption)
    } yield {
      val entries = new String(content, UTF_8)
        .parseJson
        .convertTo[Seq[MetadataEntry]]
      logger.debug(s"$name: Downloaded metadata from ${config.server.uri}: ${jsonId.id} ($length bytes) | SHA1: $sha1 | ${entries.size} entries")
      entries.map(e => e.artifactId.copy(repository = Some(name)) -> e.metadata).toMap
    }
  }

  override def fetchLogos: Future[ByteArrayInputStream] = {
    logger.debug(s"$name: Downloading logos")
    for {
      logoArtifact <- searchLatestMetadataArtifact("zip", Some("data"))

      getSha1 <- downloadSha1(logoArtifact)
      sha1Content <- getSha1
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadChecksumException(logoArtifact, status, msg))

      resp <- downloadArtifact(logoArtifact)
      content <- resp
        .onSuccess(_.asByteArray)
        .onFailure((status, msg) => DownloadArtifactException(logoArtifact, status, msg))

      (sha1, length) <- verifyPlugin(logoArtifact, sha1Content, content, resp.entity.contentLengthOption)
    } yield {
      logger.debug(s"$name: Downloaded metadata logos data from ${config.server.uri}: ${logoArtifact.id} ($length bytes) | SHA1: $sha1")
      new ByteArrayInputStream(content)
    }
  }

  def shutdown(): Future[Unit] = {
    for {
//      _ <- Http().shutdownAllConnectionPools()
      _ <- system.terminate()
    } yield ()
  }

  protected def downloadArtifact(pluginId: PluginId.Artifact): Future[HttpResponse] =
    config.server.httpRequest(getRequest(pluginId))

  protected def downloadSha1(pluginId: PluginId.Artifact): Future[HttpResponse] =
    config.server.httpRequest(getRequest(pluginId.copy(packaging = s"${pluginId.packaging}.sha1")))

  protected def getRequest(pluginId: PluginId.Artifact): HttpRequest = {
    val groupPath = pluginId.groupId.split('.').foldLeft[Uri.Path](Uri.Path.Empty)(_ / _)
    val downloadUri = config.server.uri
      .withPath(config.server.serviceUri.path ++
        (NexusPluginRepository.repositoriesPath / config.repositoryId) ++ NexusPluginRepository.contentPath ++
        (groupPath / pluginId.artifactId / pluginId.artifactVersion.id / pluginId.filename)
      )
    HttpRequest(GET,
      uri = downloadUri,
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/zip`, `application/octet-stream`, `application/json`)
      ))
  }

  protected def searchArtifact(artifact: Option[String]): Future[HttpResponse] =
    config.server.httpRequest(searchRequest(searchArtifactQuery(artifact)))

  protected def searchMetadata(packaging: String, classifier: Option[String] = None): Future[HttpResponse] = {
    config.server.httpRequest(searchRequest(searchMetadataQuery(packaging, classifier)))
  }

  protected def searchRequest(query: Uri.Query): HttpRequest = {
    HttpRequest(GET,
      uri = config.server.uri
        .withPath(config.server.serviceUri.path ++ NexusPluginRepository.searchPath)
        .withQuery(query),
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/json`)
      )
    )
  }

  protected def searchArtifactQuery(query: Option[String]): Uri.Query = paramsToQuery {
    searchParams(config.listMethod) + ("q" -> query)
  }

  protected def searchMetadataQuery(packaging: String, classifier: Option[String]): Uri.Query = paramsToQuery {
    searchParams(ListMethod.ByGroupId(
      groupId = config.metadataArtifactId.groupId,
      classifier = classifier
    )) ++ Map(
      "a" -> Some(config.metadataArtifactId.artifactId)
    )
  }

  protected def searchParams(method: ListMethod): Map[String, Option[String]] = Map("repositoryId" -> Some(config.repositoryId)) ++ {
    method match {
      case g: ByGroupId => Map(
        "g" -> Some(g.groupId),
        "c" -> g.classifier
      )
    }
  }

  def searchLatestMetadataArtifact(packaging: String, classifier: Option[String]): Future[PluginId.Artifact] = {
    logger.debug(s"$name: Searching latest version of ${config.metadataArtifactId.id}, classifier: $classifier, packaging: $packaging")
    for {
      getAllVersions <- searchMetadata(packaging, classifier)
      allVersions <- getAllVersions
        .onSuccess(_.asJson[NexusArtifactResults])
        .onFailure((status, msg) => SearchMetadataFailed(config.metadataArtifactId, packaging, classifier, status, msg))
        .map(_.data.collect {
          case r: ArtifactResult if r.allExtensions.keySet contains packaging =>
            r.version
        })

      artifact <- Future.fromTry {
        latestVersion(config.metadataVersion)(allVersions) match {
          case None =>
            logger.debug(s"$name: Could not find any candidate for metadata artifact ${config.metadataArtifactId.id}${classifier.fold("")("-" + _)}.$packaging")
            Failure(NoMetadataVersions(config.metadataArtifactId, packaging, classifier, config.metadataVersion, allVersions))
          case Some(version) =>
            Success(config.metadataArtifactId.toArtifact(name, version, packaging, classifier))
        }
      }
    } yield {
      logger.debug(s"$name: Found metadata artifact: ${artifact.id}")
      artifact
    }
  }
}

object NexusPluginRepository {
  val searchPath: Uri.Path = Uri.Path.Empty / "lucene" / "search"
  val repositoriesPath: Uri.Path = Uri.Path.Empty / "repositories"
  val contentPath: Uri.Path = Uri.Path.Empty / "content"

  val cachePath: Path = Paths.get("cache")

  def memCached(name: String, config: NexusRepositoryConfig): NexusPluginRepository = {
    new NexusPluginRepository(name, config, new PluginMetadataMemoryStorage(name))
  }

  def verifyContentLength(pluginId: PluginId.Artifact, expected: Option[Long], found: Long): Try[Long] = expected match {
    case None => Success(found)
    case Some(length) if found == length => Success(found)
    case Some(length) => Failure(ContentLengthMismatch(pluginId, found, length))
  }

  def verifySHA1(pluginId: PluginId.Artifact, expected: Array[Byte], found: Array[Byte]): Try[String] =
    for {
      sha1 <- Try(new String(expected, UTF_8))
      digest <- Try(DigestUtils.sha1Hex(found))
      result <- {
        if (digest == sha1) {
          Success(digest)
        } else {
          Failure(ChecksumMismatch(pluginId, digest, sha1))
        }
      }
    } yield result

  def verifyPlugin(pluginId: PluginId.Artifact, sha1: Array[Byte], plugin: Array[Byte], length: Option[Long]): Future[(String, Long)] = Future.fromTry {
    for {
      digest <- verifySHA1(pluginId, sha1, plugin)
      length <- verifyContentLength(pluginId, length, plugin.length)

    } yield (digest, length)
  }

  def latestVersion(pv: Version)
                   (versions: Seq[Version]): Option[Version] =
    versions
      .filter(isProductVersion(pv))
      .sortBy(_.id)
      .lastOption

  def isProductVersion(pv: Version)(v: Version): Boolean =
    pv.major == v.major && pv.minor == v.minor && pv.revision == v.revision


}

