package com.xebialabs.plugin.manager.repository.nexus

import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.metadata.Version
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.MetadataId
import org.apache.pekko.http.scaladsl.model.StatusCode

sealed abstract class NexusPluginRepositoryException(msg: String) extends DeployitException(msg)

object NexusPluginRepositoryException {

  @HttpResponseCodeResult(statusCode = 503)
  case class DownloadArtifactException(pluginId: PluginId.Artifact, statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: failed to download artifact: ${statusCode.intValue()}) $payload")

  @HttpResponseCodeResult(statusCode = 503)
  case class DownloadChecksumException(pluginId: PluginId.Artifact, statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: failed to download the SHA1 checksum: (${statusCode.intValue()}) $payload")

  @HttpResponseCodeResult(statusCode = 503)
  case class ChecksumMismatch(pluginId: PluginId.Artifact, found: String, expected: String)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: SHA1 checksum mismatch: found: $found, expected: $expected")

  @HttpResponseCodeResult(statusCode = 503)
  case class ContentLengthMismatch(pluginId: PluginId.Artifact, found: Long, expected: Long)
    extends NexusPluginRepositoryException(s"Artifact ${pluginId.id}: content-length mismatch: found: $found, expected: $expected")

  @HttpResponseCodeResult(statusCode = 503)
  case class QueryFailed(query: Option[String], statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(query.fold("List query")(q => "Searching for '$q'") ++ s" failed: (${statusCode.intValue()}) $payload")

  @HttpResponseCodeResult(statusCode = 503)
  case class SearchMetadataFailed(metadataId: MetadataId, packaging: String, classifier: Option[String], statusCode: StatusCode, payload: String)
    extends NexusPluginRepositoryException(
      s"Searching metadata version for ${metadataId.id}${classifier.fold("")("-"+ _)}.$packaging failed: (${statusCode.intValue()}) $payload"
    )

  @HttpResponseCodeResult(statusCode = 503)
  case class TooManyResults(query: Option[String], results: Int)
    extends NexusPluginRepositoryException(s"Too many results for query '${query.fold("")(q => s"$q")}': $results")

  @HttpResponseCodeResult(statusCode = 503)
  case class NoMetadataVersions(metadataId: MetadataId, packaging: String, classifier: Option[String], pv: Version, versions: Seq[Version])
    extends NexusPluginRepositoryException(
      s"No metadata versions found matching ${pv.id}. Metadata artifact ${metadataId.id}${classifier.fold("")("-"+ _)}.$packaging. Available versions: ${versions.map(_.id).mkString(",")}"
    )
}
