package com.xebialabs.plugin.manager.repository.nexus

import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.exception.PluginRepositoryException._
import com.xebialabs.plugin.manager.metadata._
import com.xebialabs.plugin.manager.repository.protocol.ArtifactResult
import com.xebialabs.plugin.manager.repository.storage.{PluginMetadataMemoryStorage, PluginMetadataStorage}
import com.xebialabs.plugin.manager.repository.{EntityOps, HttpPluginRepository, RespSyntax, latestVersion}
import grizzled.slf4j.Logging
import org.apache.pekko.http.scaladsl.model.HttpCharsets.`UTF-8`
import org.apache.pekko.http.scaladsl.model.HttpMethods._
import org.apache.pekko.http.scaladsl.model.MediaTypes.{`application/json`, `application/octet-stream`, `application/zip`}
import org.apache.pekko.http.scaladsl.model._
import org.apache.pekko.http.scaladsl.model.headers.{Accept, Authorization, `Accept-Charset`}

import java.nio.file.{Path, Paths}
import scala.concurrent.Future
import scala.util.{Failure, Success}

case class NexusPluginRepository(name: String,
                                 config: NexusRepositoryConfig,
                                 cache: PluginMetadataStorage)
  extends HttpPluginRepository(name, config, cache)
    with NexusJsonProtocol
    with MetadataEntry.Protocol
    with Logging {


  override def searchFromRemote(query: Option[String]): Future[Seq[ArtifactResult]] = {
    logger.info(s"$name: Searching for '${query.getOrElse("")}'")
    for {
      resp <- searchArtifact(query)
      content <- resp
        .onSuccess(_.asJson[NexusArtifactResults])
        .onFailure((status, msg) => QueryFailed(query, status, msg))
      plugins <- if (content.tooManyResults) {
        Future.failed(TooManyResults(query, content.totalCount))
      } else {
        Future.successful(
          content.data.filter(_.extensions.nonEmpty)
        )
      }
    } yield {
      plugins.foreach { r =>
        logger.trace(s"$name: found ${r.groupId}/${r.artifactId}-${r.version.id} ${r.extensions.keySet.mkString("(", ",", ")")}")
      }
      logger.info(s"$name: Found ${plugins.size} plugins (query: '${query.getOrElse("")}')")
      plugins
    }
  }


  protected def getRequest(pluginId: PluginId.Artifact): HttpRequest = {
    val groupPath = pluginId.groupId.split('.').foldLeft[Uri.Path](Uri.Path.Empty)(_ / _)
    val downloadUri = config.server.uri
      .withPath(config.server.serviceUri.path ++
        (NexusPluginRepository.repositoriesPath / config.repositoryId) ++ NexusPluginRepository.contentPath ++
        (groupPath / pluginId.artifactId / pluginId.artifactVersion.id / pluginId.filename)
      )
    HttpRequest(GET,
      uri = downloadUri,
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/zip`, `application/octet-stream`, `application/json`)
      ))
  }


  protected def searchRequest(query: Uri.Query): HttpRequest = {
    HttpRequest(GET,
      uri = config.server.uri
        .withPath(config.server.serviceUri.path ++ NexusPluginRepository.searchPath)
        .withQuery(query),
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/json`)
      )
    )
  }

  def searchLatestMetadataArtifact(packaging: String, classifier: Option[String]): Future[PluginId.Artifact] = {
    logger.debug(s"$name: Searching latest version of ${config.metadataArtifactId.id}, classifier: $classifier, packaging: $packaging")
    for {
      getAllVersions <- searchMetadata(packaging, classifier)
      allVersions <- getAllVersions
        .onSuccess(_.asJson[NexusArtifactResults])
        .onFailure((status, msg) => SearchMetadataFailed(config.metadataArtifactId, packaging, classifier, status, msg))
        .map(_.data.collect {
          case r: ArtifactResult if r.allExtensions.keySet contains packaging =>
            r.version
        })

      artifact <- Future.fromTry {
        latestVersion(config.metadataVersion)(allVersions) match {
          case None =>
            logger.debug(s"$name: Could not find any candidate for metadata artifact ${config.metadataArtifactId.id}${classifier.fold("")("-" + _)}.$packaging")
            Failure(NoMetadataVersions(config.metadataArtifactId, packaging, classifier, config.metadataVersion, allVersions))
          case Some(version) =>
            Success(config.metadataArtifactId.toArtifact(name, version, packaging, classifier))
        }
      }
    } yield {
      logger.debug(s"$name: Found metadata artifact: ${artifact.id}")
      artifact
    }
  }
}

object NexusPluginRepository {
  val searchPath: Uri.Path = Uri.Path.Empty / "lucene" / "search"
  val repositoriesPath: Uri.Path = Uri.Path.Empty / "repositories"
  val contentPath: Uri.Path = Uri.Path.Empty / "content"

  val cachePath: Path = Paths.get("cache")

  def memCached(name: String, config: NexusRepositoryConfig): NexusPluginRepository = {
    new NexusPluginRepository(name, config, new PluginMetadataMemoryStorage(name))
  }

}

