package com.xebialabs.plugin.manager.startup

import com.xebialabs.plugin.manager.compatibility.{Jdk17Compatibility, Jdk17ReportUtils}
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.model.{DbPlugin, FilePlugin}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.rest.api.PluginSource.LOCAL
import com.xebialabs.plugin.manager.rest.api.PluginStatus
import com.xebialabs.plugin.manager.util.PluginsTraverser.{listOfficialFsPlugins, traverse}
import grizzled.slf4j.Logging

import java.nio.file.{Path, Paths}
import scala.util.Try

/**
  * - upgrades existing plugins in the DB if there's a newer version on the FS
  * - inserts plugins that exist on FS and not in DB
  *
  * Intended to be run on product upgrade.
  */
class PluginUpgrader(val pluginRepository: SqlPluginRepository, val product: String, val pluginsDir: Path) extends Logging{

  def this(pluginRepository: SqlPluginRepository, product: String) = {
    this(pluginRepository, product, Paths.get("plugins"))
  }

  val xlProduct: XLProduct = XLProduct.fromString(product)

  def upgradePlugins(): Boolean = {
    upgradePlugins(false)
  }

  def upgradePlugins(checkJdk17Compatibility: Boolean): Boolean =  {
    val dbPluginsBySource = pluginRepository.getAllWithBytes.groupBy(_.source.toString)

    traverse(pluginsDir)(
      (officialSource, plugins) => upgradeOfficial(plugins, dbPluginsBySource.getOrElse(officialSource.toString, Seq.empty)),
      plugins => upgradeLocal(plugins, dbPluginsBySource.getOrElse(LOCAL.toString, Seq.empty)),
    )

    pluginRepository.updateAllPluginsStatusTo(PluginStatus.INSTALLED)

    if (checkJdk17Compatibility) {
      val dbplugins = pluginRepository.getAllWithBytes
      val fsOfficialPlugins = listOfficialFsPlugins(pluginsDir)
      val dbPluginsFiltered = dbplugins.filter(_.doesntExistIn(fsOfficialPlugins))
      val pluginCompatibility = dbPluginsFiltered.map(plugin => plugin.name -> Try(Jdk17Compatibility.isCompatible(plugin)))

      logger.info(Jdk17ReportUtils.displayCompatibilityReport(pluginCompatibility.iterator.toMap)(new StringBuffer()))

      pluginCompatibility.forall(b => b._2.isSuccess && b._2.get)
    } else {
      true
    }
  }

  private def upgradeOfficial(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    debug(s"Processing list of official plugins found on the filesystem $fsPlugins and in the database $dbPlugins")
    // cleanup old versions of bundled plugins
    dbPlugins.filter(_.higherVersionExistsIn(fsPlugins)).foreach(dbPlugin => cleanupFromDatabase(dbPlugin))

    // for pre-10.2.0 to current version upgrade scenario. All plugins from pre-10.2.0 version will end up in plugins
    // folder of the current version after setup with -previous-installation.
    fsPlugins.filter(fsPlugin => fsPlugin.doesntExistIn(dbPlugins) || fsPlugin.isHigherVersionThanAMatchIn(dbPlugins))
      .foreach(fsPlugin => writeOfficialToDatabase(fsPlugin))
  }

  private def upgradeLocal(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    debug(s"Processing list of local plugins found on the filesystem $fsPlugins and in the database $dbPlugins")

    // for pre-10.2.0 to current version upgrade scenario. All plugins from pre-10.2.0 version will end up in plugins
    // folder of the current version after setup with -previous-installation.
    dbPlugins.filter(_.existsIn(fsPlugins)).foreach(cleanupFromDatabase)
    fsPlugins.foreach(fsPlugin => writeLocalToDatabase(fsPlugin))
  }

  private def cleanupFromDatabase(dbPlugin: DbPlugin): Unit = {
    info(s"Removing plugin ${dbPlugin.name} from database")
    pluginRepository.delete(dbPlugin)
  }

  private def writeOfficialToDatabase(fsPlugin: FilePlugin): Unit = {
    info(s"Synchronizing plugin ${fsPlugin.name} from filesystem to database")
    pluginRepository.insert(fsPlugin.toOfficialDbPlugin(xlProduct))
    fsPlugin
  }

  private def writeLocalToDatabase(fsPlugin: FilePlugin): Unit = {
    info(s"Synchronizing plugin ${fsPlugin.name} from filesystem to database")
    pluginRepository.insert(fsPlugin.toLocalDbPlugin)
  }

}
