package com.xebialabs.plugin.manager.util

import com.xebialabs.deployit.util.TFiles
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.exception.{InvalidPluginVersionPropertiesException, MissingPluginVersionPropertiesException}
import com.xebialabs.plugin.manager.metadata.PluginVersionProperties
import com.xebialabs.plugin.protocol.xlp.JarURL
import de.schlichtherle.truezip.file.TFile
import org.springframework.core.io.UrlResource
import org.springframework.core.io.support.PropertiesLoaderUtils
import org.springframework.util.FileSystemUtils
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import com.xebialabs.plugin.classloader.PluginClassLoader
import com.xebialabs.plugin.manager.model.DbPlugin

import java.io.{File, IOException}
import java.net.{JarURLConnection, URLConnection}
import java.nio.file.{Files, Path}
import java.util.zip.{ZipException, ZipFile}
import scala.util.Using

object PluginFileUtils extends Logging{

  def getPluginExtension(filename: String): String = {
    ConfigWrapper.getValidExtension(filename) match {
      case Some(extension) => extension
      case None => throw new IllegalArgumentException(s"Filename must end with '${ConfigWrapper.getExtensions().map(ext => s".${ext}").mkString(" or ")}'")
    }
  }

  def getPluginPropertiesForZip(filePath: Path): PluginVersionProperties = {
    getPluginPropertiesForZip(filePath.toFile)
  }

  def getPluginPropertiesForZip(filename: String, bytes: Array[Byte]): PluginVersionProperties = {
    withTmpDir { tmpDir =>
      val file: File = getTemporaryPluginFile(tmpDir, filename, bytes)
      getPluginPropertiesForZip(file)
    }
  }

  def getPluginPropertiesForZip(file: File): PluginVersionProperties = {
    try {
      val filename: String = file.getName
      Using.resource(new ZipFile(file)) { zipfile =>
        val urls = Option(zipfile.getEntry("plugin-version.properties")) match {
          case Some(entry) => Seq(JarURL(file.getAbsolutePath, entry.getName))
          case None => throw MissingPluginVersionPropertiesException(s"Could not read plugin-version.properties in $filename")
        }
        val props = PropertiesLoaderUtils.loadProperties(new UrlResource(urls.head) {
          override def customizeConnection(con: URLConnection): Unit = {
            super.customizeConnection(con)
            if (con.isInstanceOf[JarURLConnection]) {
              con.setUseCaches(false)
            }
          }
        })
        val name = Option(props.getProperty("plugin"))
        val version = Option(props.getProperty("version"))
        val pluginType = Option(props.getProperty("type"))

        if (name.isDefined && version.isDefined) {
          PluginVersionProperties(name.get, version.get, pluginType)
        } else {
          throw InvalidPluginVersionPropertiesException(s"$filename does not have a valid plugin-version.properties")
        }
      }
    } catch {
      case e: IOException => throw new IllegalStateException(e)
      case e: ZipException => throw new IllegalStateException(e)
    }
  }

  private def withTmpDir[R](block: Path => R): R = {
    val tmpDir = Files.createTempDirectory("plugin")
    try {
      block(tmpDir)
    } finally {
      FileSystemUtils.deleteRecursively(tmpDir)
    }
  }

  private def getTemporaryPluginFile(tmpDir: Path, filename: String, bytes: Array[Byte]): File = {
    val pluginPath = tmpDir.resolve(filename)
    Files.write(pluginPath, bytes).toFile
  }

  def deletePluginsFromFilesystem[T](
                                      plugins: Seq[T],
                                      getPluginFile: T => java.nio.file.Path,
                                      getPluginName: T => String,
                                      getPluginVersion: T => Option[String] = (_: T) => None
                                    ): Unit = {
    val classLoader = Thread.currentThread.getContextClassLoader

    def deleteFile(pluginFile: java.nio.file.Path): Unit = {
      if (Files.exists(pluginFile)) {
        TFiles.umountQuietly(new TFile(pluginFile.toFile))
        FileUtils.forceDelete(pluginFile.toFile)
      } else {
        logger.warn(s"Plugin file $pluginFile does not exist, cannot delete.")
      }
    }

    classLoader match {
      case pluginClassLoader: PluginClassLoader =>
        try {
          pluginClassLoader.clearClasspathRoots()
          plugins.foreach { plugin =>
            val pluginFile = getPluginFile(plugin)
            val name = getPluginName(plugin)
            val versionStr = getPluginVersion(plugin).map(v => s" version $v").getOrElse("")
            logger.info(s"Removing plugin $name$versionStr from filesystem")
            deleteFile(pluginFile)
          }
        } catch {
          case e: Exception =>
            logger.error("Error while deleting plugins", e)
        } finally {
          pluginClassLoader.refreshDirs()
        }
      case _ =>
        plugins.foreach { plugin =>
          val pluginFile = getPluginFile(plugin)
          val name = getPluginName(plugin)
          val versionStr = getPluginVersion(plugin).map(v => s" version $v").getOrElse("")
          logger.info(s"Removing plugin $name$versionStr (No classloader) from filesystem")
          deleteFile(pluginFile)
        }
    }
  }

  def getPluginFileAndName(plugin: DbPlugin, pluginsDir: java.nio.file.Path): java.nio.file.Path = {
    val fileName = s"${plugin.name}${plugin.version.map(v => s"-$v").getOrElse("")}.${plugin.extension}"
    pluginsDir.resolve(plugin.source.toString).resolve(fileName)
  }
}