package com.xebialabs.plugin.manager.repository.artifactory

import com.xebialabs.plugin.manager.PluginId
import com.xebialabs.plugin.manager.exception.PluginRepositoryException._
import com.xebialabs.plugin.manager.metadata.{MetadataEntry, Version}
import com.xebialabs.plugin.manager.repository.artifactory.ArtifactoryArtifactResult.extractArtifactInfo
import com.xebialabs.plugin.manager.repository.protocol.ArtifactResult
import com.xebialabs.plugin.manager.repository.storage.{PluginMetadataMemoryStorage, PluginMetadataStorage}
import com.xebialabs.plugin.manager.repository.{EntityOps, HttpPluginRepository, RespSyntax, latestVersion}
import grizzled.slf4j.Logging
import org.apache.pekko.http.scaladsl.model.HttpCharsets.`UTF-8`
import org.apache.pekko.http.scaladsl.model.HttpMethods.GET
import org.apache.pekko.http.scaladsl.model.MediaTypes.{`application/json`, `application/octet-stream`, `application/zip`}
import org.apache.pekko.http.scaladsl.model.headers.{Accept, Authorization, RawHeader, `Accept-Charset`}
import org.apache.pekko.http.scaladsl.model.{HttpRequest, Uri}

import scala.concurrent.Future
import scala.util.{Failure, Success}


case class ArtifactoryPluginRepository(name: String,
                                       config: ArtifactoryRepositoryConfig,
                                       cache: PluginMetadataStorage)
  extends HttpPluginRepository(name, config, cache)
    with ArtifactoryJsonProtocol
    with MetadataEntry.Protocol
    with Logging {


  override def searchFromRemote(query: Option[String]): Future[Seq[ArtifactResult]] = {
    logger.info(s"$name: Searching for '${query.getOrElse("")}'")
    for {
      resp <- searchArtifact(query)
      content <- resp
        .onSuccess(_.asJson[ArtifactoryArtifactResults])
        .onFailure((status, msg) => QueryFailed(query, status, msg))
      plugins <- {
        Future.successful(
          content.results.filter(_.ext.nonEmpty)
        )
      }
    } yield {
      logger.info(s"$name: Found ${plugins.size} plugins (query: '${query.getOrElse("")}')")
      plugins.flatMap(plugin => extractArtifactInfo(plugin))
    }
  }


  protected def getRequest(pluginId: PluginId.Artifact): HttpRequest = {
    val groupPath = pluginId.groupId.split('.').foldLeft[Uri.Path](Uri.Path.Empty)(_ / _)
    val downloadUri = config.server.uri
      .withPath(config.server.serviceUri.path / config.repositoryId ++
        groupPath / pluginId.artifactId / pluginId.artifactVersion.id / pluginId.filename
      )
    HttpRequest(GET,
      uri = downloadUri,
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/zip`, `application/octet-stream`, `application/json`)
      ))
  }


  protected def searchRequest(query: Uri.Query): HttpRequest = {
    HttpRequest(GET,
      uri = config.server.uri
        .withPath(config.server.serviceUri.path ++ ArtifactoryPluginRepository.searchPath)
        .withQuery(query),
      headers = List(
        Authorization(config.server.credentials),
        `Accept-Charset`(`UTF-8`),
        Accept(`application/json`),
        RawHeader("X-Result-Detail", "info")
      )
    )
  }


  def searchLatestMetadataArtifact(packaging: String, classifier: Option[String]): Future[PluginId.Artifact] = {
    logger.debug(s"$name: Searching latest version of ${config.metadataArtifactId.id}, classifier: $classifier, packaging: $packaging")
    for {
      getAllVersions <- searchMetadata(packaging, classifier)
      allVersions <- getAllVersions
        .onSuccess(_.asJson[ArtifactoryArtifactResults])
        .onFailure((status, msg) => SearchMetadataFailed(config.metadataArtifactId, packaging, classifier, status, msg))
        .map(result => {
          result.results.collect {
            case r: ArtifactoryArtifactResult if r.ext contains packaging =>
              Version.fromString(r.version)
          }.flatten
        })

      artifact <- Future.fromTry {
        latestVersion(config.metadataVersion)(allVersions) match {
          case None =>
            logger.debug(s"$name: Could not find any candidate for metadata artifact ${config.metadataArtifactId.id}${classifier.fold("")("-" + _)}.$packaging")
            Failure(NoMetadataVersions(config.metadataArtifactId, packaging, classifier, config.metadataVersion, allVersions))
          case Some(version) =>
            Success(config.metadataArtifactId.toArtifact(name, version, packaging, classifier))
        }
      }
    } yield {
      logger.debug(s"$name: Found metadata artifact: ${artifact.id()}")
      artifact
    }
  }

}

object ArtifactoryPluginRepository {
  val searchPath: Uri.Path = Uri.Path.Empty / "api" / "search" / "gavc"

  def memCached(name: String, config: ArtifactoryRepositoryConfig): ArtifactoryPluginRepository = {
    new ArtifactoryPluginRepository(name, config, new PluginMetadataMemoryStorage(name))
  }

}
