package com.xebialabs.plugin.manager.metadata

import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._


sealed trait XLProduct

object XLProduct {
  case object XLRelease extends XLProduct
  case object XLDeploy extends XLProduct

  def fromString(productName: String): XLProduct = productName match {
    case "xl-release" => XLRelease
    case "xl-deploy" => XLDeploy
    case other => throw new IllegalArgumentException(s"Unknown product name: '$other'")
  }

  def id(product: XLProduct): String = product match {
    case XLRelease => "xl-release"
    case XLDeploy => "xl-deploy"
  }

  def short(product: XLProduct): String = product match {
    case XLRelease => "xlr"
    case XLDeploy => "xld"
  }

  implicit class XLProductOps(val product: XLProduct) extends AnyVal {
    def id: String = XLProduct.id(product)
    def short: String = XLProduct.short(product)
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {
    val xlProductString: XLProduct => String = {
      case XLRelease => "XL Release"
      case XLDeploy => "XL Deploy"
    }
    val xlProductWriter: JsonWriter[XLProduct] = p => xlProductString(p).toJson
    val xlProductReader: JsonReader[XLProduct] = jsonReader[XLProduct] {
      case JsString("XL Release") => XLRelease
      case JsString("XL Deploy") => XLDeploy
      case unknown =>
        deserializationError(s"Unknown XL product: '$unknown'")
    }
    implicit val xlProductFormat: JsonFormat[XLProduct] = jsonFormat(xlProductReader, xlProductWriter)
  }
}