package com.xebialabs.plugin.manager.startup

import com.xebialabs.plugin.manager.compatibility.{Jdk17Compatibility, Jdk17ReportUtils}
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.model.{DbPlugin, FilePlugin}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.rest.api.PluginSource.LOCAL
import com.xebialabs.plugin.manager.rest.api.PluginStatus
import com.xebialabs.plugin.manager.util.PluginsTraverser.{listOfficialFsPlugins, traverse}
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import com.github.zafarkhaja.semver.Version
import com.xebialabs.plugin.manager.util.PluginFileUtils

import java.io.File
import java.nio.file.{Path, Paths}
import scala.util.Try

/**
 * - upgrades existing plugins in the DB if there's a newer version on the FS
 * - inserts plugins that exist on FS and not in DB
 *
 * Intended to be run on product upgrade.
 */
class PluginUpgrader(val pluginRepository: SqlPluginRepository, val product: String, val pluginsDir: Path) extends Logging{

  def this(pluginRepository: SqlPluginRepository, product: String) = {
    this(pluginRepository, product, Paths.get("plugins"))
  }

  val xlProduct: XLProduct = XLProduct.fromString(product)

  def upgradePlugins(): Boolean = {
    upgradePlugins(false)
  }

  def upgradePlugins(checkJdk17Compatibility: Boolean): Boolean =  {
    val dbPluginsBySource = pluginRepository.getAllWithBytes.groupBy(_.source.toString)

    traverse(pluginsDir)(
      (officialSource, plugins) => upgradeOfficial(plugins, dbPluginsBySource.getOrElse(officialSource.toString, Seq.empty)),
      plugins => upgradeLocal(plugins, dbPluginsBySource.getOrElse(LOCAL.toString, Seq.empty)),
    )

    cleanupDuplicatePlugins()

    pluginRepository.updateAllPluginsStatusTo(PluginStatus.INSTALLED)

    if (checkJdk17Compatibility) {
      val dbplugins = pluginRepository.getAllWithBytes
      val fsOfficialPlugins = listOfficialFsPlugins(pluginsDir)
      val dbPluginsFiltered = dbplugins.filter(_.doesntExistIn(fsOfficialPlugins))
      val pluginCompatibility = dbPluginsFiltered.map(plugin => plugin.name -> Try(Jdk17Compatibility.isCompatible(plugin)))

      logger.info(Jdk17ReportUtils.displayCompatibilityReport(pluginCompatibility.iterator.toMap)(new StringBuffer()))

      pluginCompatibility.forall(b => b._2.isSuccess && b._2.get)
    } else {
      true
    }
  }

  private def cleanupDuplicatePlugins(): Unit = {
    val dbPluginsByName = pluginRepository.getAllWithBytes.groupBy(_.name)
    var pluginsToRemove: Seq[DbPlugin] = Seq.empty
    dbPluginsByName.foreach { case (pluginName, plugins) =>
      if (plugins.size > 1) {
        logger.info(s"Found duplicate plugins with name: $pluginName. Number of duplicates: ${plugins.size}")
        pluginsToRemove = handleDuplicatePlugin(plugins, pluginsToRemove, pluginName)
      }
    }
    if (pluginsToRemove.nonEmpty) {
      deleteOldPluginsFromFilesystem(pluginsToRemove)
    }
  }

  private def handleDuplicatePlugin(plugins: Seq[DbPlugin], pluginsToRemove: Seq[DbPlugin], pluginName: String): Seq[DbPlugin] = {
    val sortedPlugins = plugins.sortBy(p => Version.valueOf(p.version.getOrElse("0.0.0")))(Ordering.by((v: Version) => v).reverse)
    val (latestPlugin, oldPlugin) =
      if ((sortedPlugins.tail.last.version == sortedPlugins.head.version) && sortedPlugins.head.source == LOCAL) {
        // If versions are the same, prefer the official plugin as latest
        (sortedPlugins.tail.last, sortedPlugins.head)
      } else {
        (sortedPlugins.head, sortedPlugins.tail.last)
      }
    logger.info(s"Duplicate plugin identified: $pluginName. Keeping latest version: ${latestPlugin.version.getOrElse("No version specified")}")
    logger.info(s"Removing old plugin version from database: ${oldPlugin.version.getOrElse("No version specified")}")
    cleanupFromDatabase(oldPlugin)

    if (latestPlugin.source == LOCAL) {
      logger.info(s"Writing latest plugin to filesystem")
      writeToFilesystem(latestPlugin, oldPlugin.source.toString)
      logger.info(s"Sync the official database details between old and new plugin.")
      val updatedPlugin = latestPlugin.copy(
        groupId = oldPlugin.groupId,
        source = oldPlugin.source
      )
      pluginRepository.update(updatedPlugin, latestPlugin)
    }
    logger.info(s"Adding old plugin to list to deleting: ${oldPlugin.name} version " +
      s"${oldPlugin.version.getOrElse("No version specified")}")
    pluginsToRemove :+ oldPlugin
  }

  private def deleteOldPluginsFromFilesystem(removablePluginSeq: Seq[DbPlugin]): Unit = {
    PluginFileUtils.deletePluginsFromFilesystem[DbPlugin](
      removablePluginSeq,
      plugin => PluginFileUtils.getPluginFileAndName(plugin, pluginsDir),
      plugin => plugin.name,
      plugin => plugin.version
    )
  }

  private def writeToFilesystem(dbPlugin: DbPlugin, source: String): Unit = {
    info(s"Copying plugin ${dbPlugin.name} from database to filesystem")
    val pluginSource = source
    val fileName = s"${dbPlugin.name}${if (dbPlugin.version.isEmpty) "" else "-" + dbPlugin.version.get}.${dbPlugin.extension}"
    val targetFile = new File(s"$pluginsDir${File.separator}${pluginSource}${File.separator}$fileName")
    if (!targetFile.exists()) {
      FileUtils.writeByteArrayToFile(targetFile, dbPlugin.bytes.get)
    } else {
      info(s"File $fileName already exists, skipping write.")
    }
  }

  private def upgradeOfficial(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    debug(s"Processing list of official plugins found on the filesystem $fsPlugins and in the database $dbPlugins")
    // cleanup old versions of bundled plugins
    dbPlugins.filter(_.higherVersionExistsIn(fsPlugins)).foreach(dbPlugin => cleanupFromDatabase(dbPlugin))

    // for pre-10.2.0 to current version upgrade scenario. All plugins from pre-10.2.0 version will end up in plugins
    // folder of the current version after setup with -previous-installation.
    fsPlugins.filter(fsPlugin => fsPlugin.doesntExistIn(dbPlugins) || fsPlugin.isHigherVersionThanAMatchIn(dbPlugins))
      .foreach(fsPlugin => writeOfficialToDatabase(fsPlugin))
  }

  private def upgradeLocal(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    debug(s"Processing list of local plugins found on the filesystem $fsPlugins and in the database $dbPlugins")

    // for pre-10.2.0 to current version upgrade scenario. All plugins from pre-10.2.0 version will end up in plugins
    // folder of the current version after setup with -previous-installation.
    dbPlugins.filter(_.existsIn(fsPlugins)).foreach(cleanupFromDatabase)
    fsPlugins.foreach(fsPlugin => writeLocalToDatabase(fsPlugin))
  }

  private def cleanupFromDatabase(dbPlugin: DbPlugin): Unit = {
    try {
      info(s"Removing plugin ${dbPlugin.name} from database")
      pluginRepository.delete(dbPlugin)
    } catch {
      case e: Exception =>
        logger.error(s"Failed to remove plugin ${dbPlugin.name} from database", e)
    }
  }

  private def writeOfficialToDatabase(fsPlugin: FilePlugin): Unit = {
    info(s"Synchronizing plugin ${fsPlugin.name} from filesystem to database")
    pluginRepository.insert(fsPlugin.toOfficialDbPlugin(xlProduct))
    fsPlugin
  }

  private def writeLocalToDatabase(fsPlugin: FilePlugin): Unit = {
    info(s"Synchronizing plugin ${fsPlugin.name} from filesystem to database")
    pluginRepository.insert(fsPlugin.toLocalDbPlugin)
  }

}