package com.xebialabs.plugin.manager.metadata

import com.xebialabs.plugin.manager.requireProperFilename
import org.apache.pekko.http.scaladsl.marshallers.sprayjson.SprayJsonSupport
import spray.json._

import scala.util.matching.Regex

case class Version private(major: Int, minor: Option[Int], revision: Option[Int], extra: Option[String]) {
  require(major >= 0, "version major can't be negative")
  require(minor.fold(true)(_ >= 0), "version minor can't be negative")
  require(revision.isEmpty || (revision.nonEmpty && minor.nonEmpty), "version revision requires version minor")
  revision.foreach(r => require(r >= 0, "version revision can't be negative"))
  extra.foreach(requireProperFilename(_, "extra"))
}

object Version {

  def fromPlatform(version: com.xebialabs.deployit.server.api.upgrade.Version): Version = {
    val extra = (Option(version.getDataModel), Option(version.getClassifier)) match {
      case (None, None) => None
      case (Some(dm), None) => Some(s"#$dm")
      case (None, Some(classifier)) => Some(s"-$classifier")
      case (Some(dm), Some(classifier)) => Some(s"#$dm-$classifier")
    }
    new Version(version.getMajor, Option(version.getMinor), Option(version.getMicro), extra)
  }

  val zero = Version(0)

  val versionPattern: Regex = """(\d+)(\.\d+)?(\.\d+)?(.*)?""".r

  def apply(major: Int): Version = new Version(major, None, None, None)

  def apply(major: Int, minor: Int) = new Version(major, Some(minor), None, None)

  def apply(major: Int, minor: Int, revision: Int): Version = new Version(major, Some(minor), Some(revision), None)

  def apply(major: Int, minor: Int, revision: Int, extra: Option[String]): Version =
    new Version(major, Some(minor), Some(revision), extra.filter(_.nonEmpty))

  def apply(major: Int, minor: Option[Int] = None, revision: Option[Int] = None, extra: Option[String] = None): Version = {
    new Version(major, minor, revision, extra.filter(_.nonEmpty))
  }

  def fromString(version: String): Option[Version] = version match {
    case versionPattern(major, minor, revision, extra) =>
      Some(
        Version(
          major = Integer.parseInt(major),
          minor = Option(minor).map(v => Integer.parseInt(v.tail)),
          revision = Option(revision).map(v => Integer.parseInt(v.tail)),
          extra = Option(extra).filter(_.nonEmpty)
        )
      )
    case _ => None
  }

  implicit class VersionOps(val version: Version) extends AnyVal {
    def id: String = {
      s"${version.major}" ++
        version.minor.fold("")(minor => s".$minor" ++ version.revision.fold("")(revision => s".$revision")) ++
        version.extra.getOrElse("")
    }
  }

  trait Protocol extends SprayJsonSupport with DefaultJsonProtocol {

    val versionWriter: JsonWriter[Version] = v => v.id.toJson
    val versionReader: JsonReader[Version] = jsonReader[Version] {
      case JsString(versionString) =>
        fromString(versionString) match {
          case None =>
            deserializationError(s"Cannot parse version string: '$versionString'")
          case Some(version) => version
        }
      case unknown =>
        deserializationError(s"Cannot parse version: '$unknown'")
    }
    implicit val versionConstantFormat: JsonFormat[Version] = jsonFormat(versionReader, versionWriter)
  }

}
