package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.engine.spi.command.*;
import com.xebialabs.deployit.engine.spi.command.util.Update;
import com.xebialabs.deployit.engine.spi.event.DeployitEventListener;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.core.Directory;

import nl.javadude.t2bus.Subscribe;

import java.util.List;
import java.util.stream.Collectors;

@DeployitEventListener
public class RepositoryEventListener {

    public static final String DUPLICATE_NAME_SUFFIX = "Copy";

    @Subscribe
    public void receiveCreate(CreateCiCommand command) {
        checkApplicationOnUniqueness(command.getCi());
        RepositoryServiceHolder.getRepositoryService().create(command.getCi());
    }

    @Subscribe
    public void receiveCreate(CreateCisCommand command) {
        final List<ConfigurationItem> listOfCis = command.getCis();
        ConfigurationItem[] cis = listOfCis.toArray(new ConfigurationItem[listOfCis.size()]);
        checkApplicationOnUniqueness(cis);
        RepositoryServiceHolder.getRepositoryService().create(cis);
    }

    @Subscribe
    public void receiveUpdate(UpdateCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().update(command.getUpdate().getNewCi());
    }

    @Subscribe
    public void receiveUpdate(UpdateCisCommand command) {
        List<Update> updates = command.getUpdates();
        List<ConfigurationItem> collect = updates.stream().map(Update::getNewCi).collect(Collectors.toList());
        RepositoryServiceHolder.getRepositoryService().createOrUpdate(collect.toArray(new ConfigurationItem[collect.size()]));
    }

    @Subscribe
    public void receiveDelete(DeleteCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().delete(command.getCiId());
    }

    @Subscribe
    public void receiveDelete(DeleteCisCommand command) {
        RepositoryServiceHolder.getRepositoryService().delete(command.getCis().toArray(new String[command.getCis().size()]));
    }

    @Subscribe
    public void receiveMove(MoveCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().move(command.getId(), command.getTargetId());
    }

    @Subscribe
    public void receiveCopy(CopyCiCommand command) {
        checkApplicationOnUniquenessByName(getName(command.getTargetId()), command.getType());
        RepositoryServiceHolder.getRepositoryService().copy(command.getId(), command.getTargetId());
        ensureChildApplicationNamesAreUnique(command);
    }

    @Subscribe
    public void receiveRename(RenameCiCommand command) {
        checkApplicationOnUniquenessByName(command.getTargetName(), command.getType());
        RepositoryServiceHolder.getRepositoryService().rename(command.getId(), command.getTargetName());
    }

    private void checkApplicationOnUniqueness(ConfigurationItem... cis) {
        for (ConfigurationItem ci : cis) {
            checkApplicationOnUniquenessByName(ci.getName(), ci.getType());
        }
    }

    private void checkApplicationOnUniquenessByName(String ciName, Type type) {
        Type applicationType = Type.valueOf(Application.class);
        if (!applicationType.instanceOf(type)) {
            return;
        }
        if (existsByName(ciName, type)) {
            throw new ItemAlreadyExistsException("Application with name [%s] already exists", ciName);
        }
    }

    private void ensureChildApplicationNamesAreUnique(final CopyCiCommand command) {
        if (!Type.valueOf(Directory.class).instanceOf(command.getType())) {
            return;
        }
        Type applicationType = Type.valueOf(Application.class);
        SearchParameters parameters = new SearchParameters().setType(applicationType).setAncestor(command.getTargetId());
        for (ConfigurationItemData childApplication : RepositoryServiceHolder.getRepositoryService().list(parameters)) {
            String duplicateName = findUniqueName(applicationType, getName(childApplication.getId()), DUPLICATE_NAME_SUFFIX);
            RepositoryServiceHolder.getRepositoryService().rename(childApplication.getId(), duplicateName);
        }
    }

    private String findUniqueName(Type type, String baseName, String suffix) {
        String name = baseName + " " + suffix;
        return existsByName(name, type) ? findUniqueName(type, name, suffix) : name;
    }

    private boolean existsByName(final String ciName, final Type type) {
        SearchParameters parameters = new SearchParameters().setType(type).setName(ciName);
        return !RepositoryServiceHolder.getRepositoryService().list(parameters).isEmpty();
    }

    private String getName(String id) {
        int indexOfLastSlash = id.lastIndexOf('/');
        if (indexOfLastSlash > -1) {
            return id.substring(indexOfLastSlash + 1);
        }
        return id;
    }
}
