package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.creator.CreatorContextImpl;
import com.xebialabs.deployit.engine.spi.command.*;
import com.xebialabs.deployit.engine.spi.command.util.Update;
import com.xebialabs.deployit.engine.spi.event.DeployitEventListener;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.core.Directory;
import nl.javadude.t2bus.Subscribe;

import java.util.List;
import java.util.function.Function;

@DeployitEventListener
public class RepositoryEventListener {

    public static final String DUPLICATE_NAME_SUFFIX = "Copy";

    @Subscribe
    public void receiveCreate(CreateCiCommand command) {
        CiRepositoryUtils.checkApplicationOnUniqueness(command.getCi());
        ChangeSet changeSet = new ChangeSet();
        populateChangeSet(command.getCi(), changeSet);
        RepositoryServiceHolder.getRepositoryService().execute(changeSet);
    }

    @Subscribe
    public void receiveCreate(CreateCisCommand command) {
        final List<ConfigurationItem> listOfCis = command.getCis();
        ConfigurationItem[] cis = listOfCis.toArray(new ConfigurationItem[listOfCis.size()]);
        CiRepositoryUtils.checkApplicationOnUniqueness(cis);
        ChangeSet changeSet = new ChangeSet();
        for (ConfigurationItem ci: cis) {
            populateChangeSet(ci, changeSet);
        }
        RepositoryServiceHolder.getRepositoryService().execute(changeSet);
    }

    @Subscribe
    public void receiveUpdate(UpdateCiCommand command) {
        ChangeSet changeSet = new ChangeSet();
        if (command.getUpdate().getPreviousCi() == null) {
            populateChangeSet(command.getUpdate().getNewCi(), changeSet, (ci) -> {
                changeSet.update(ci);
                return null;
            });
        } else {
            changeSet.update(command.getUpdate().getNewCi());
        }
        RepositoryServiceHolder.getRepositoryService().execute(changeSet);
    }

    @Subscribe
    public void receiveUpdate(UpdateCisCommand command) {
        List<Update> updates = command.getUpdates();
        ChangeSet changeSet = new ChangeSet();
        changeSet.setSCMTraceabilityData(command.getSCMTraceabilityData());
        updates.forEach(update -> {
            if (update.getPreviousCi() == null) {
                populateChangeSet(update.getNewCi(), changeSet, (ci) -> {
                    changeSet.createOrUpdate(ci);
                    return null;
                });
            } else {
                changeSet.createOrUpdate(update.getNewCi());
            }
        });
        RepositoryServiceHolder.getRepositoryService().execute(changeSet);
    }

    @Subscribe
    public void receiveDelete(DeleteCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().delete(command.getCiId());
    }

    @Subscribe
    public void receiveDelete(DeleteCisCommand command) {
        RepositoryServiceHolder.getRepositoryService().delete(command.getCis().toArray(new String[command.getCis().size()]));
    }

    @Subscribe
    public void receiveMove(MoveCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().move(command.getId(), command.getTargetId());
    }

    @Subscribe
    public void receiveCopy(CopyCiCommand command) {
        CiRepositoryUtils.checkApplicationOnUniquenessByName(getName(command.getTargetId()), command.getType());
        RepositoryServiceHolder.getRepositoryService().copy(command.getId(), command.getTargetId());
        ensureChildApplicationNamesAreUnique(command);
    }

    @Subscribe
    public void receiveRename(RenameCiCommand command) {
        CiRepositoryUtils.checkApplicationOnUniquenessByName(command.getTargetName(), command.getType());
        RepositoryServiceHolder.getRepositoryService().rename(command.getId(), command.getTargetName());
    }


    private void populateChangeSet(ConfigurationItem ci, ChangeSet changeSet) {
        populateChangeSet(ci, changeSet, (item) -> {
            changeSet.create(item);
            return null;
        });
    }

    private void populateChangeSet(ConfigurationItem ci, ChangeSet changeSet, Function<ConfigurationItem, Void> operation) {
        if (ci.getType().getDescriptor().getCreator() != null) {
            ci.getType().getDescriptor().getCreator().invoke(new CreatorContextImpl(ci, changeSet));
        } else {
            operation.apply(ci);
        }
    }

    private void ensureChildApplicationNamesAreUnique(final CopyCiCommand command) {
        if (!Type.valueOf(Directory.class).instanceOf(command.getType())) {
            return;
        }
        Type applicationType = Type.valueOf(Application.class);
        SearchParameters parameters = new SearchParameters().setType(applicationType).setAncestor(command.getTargetId());
        for (ConfigurationItemData childApplication : RepositoryServiceHolder.getRepositoryService().list(parameters)) {
            String duplicateName = CiRepositoryUtils.findUniqueName(applicationType, getName(childApplication.getId()), DUPLICATE_NAME_SUFFIX);
            RepositoryServiceHolder.getRepositoryService().rename(childApplication.getId(), duplicateName);
        }
    }

    private String getName(String id) {
        int indexOfLastSlash = id.lastIndexOf('/');
        if (indexOfLastSlash > -1) {
            return id.substring(indexOfLastSlash + 1);
        }
        return id;
    }

}
