package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.service.externalproperties.ExternalPropertiesResolver;
import com.xebialabs.deployit.service.validation.Validator;

import java.util.Arrays;
import java.util.List;

public class RepositoryAdapter implements Repository {
    private RepositoryService repositoryService;
    private Validator validator;
    private ExternalPropertiesResolver externalPropertiesResolver;
    private WorkDir workDir;

    public RepositoryAdapter(RepositoryService repositoryService, Validator validator, ExternalPropertiesResolver externalPropertiesResolver, WorkDir workDir) {
        this.repositoryService = repositoryService;
        this.validator = validator;
        this.externalPropertiesResolver = externalPropertiesResolver;
        this.workDir = workDir;
    }

    @Override
    public boolean exists(String id) {
        return repositoryService.exists(id);
    }

    @Override
    public <T extends ConfigurationItem> T read(String id) {
        T ci = repositoryService.read(id, workDir);
        externalPropertiesResolver.resolveExternalProperties(ci);
        return ci;
    }

    @Override
    public <T extends ConfigurationItem> List<T> read(List<String> ids, Integer depth) {
        List<T> cis = repositoryService.read(ids, depth, true, true);
        cis.forEach(externalPropertiesResolver::resolveExternalProperties);
        return cis;
    }

    @SafeVarargs
    @Override
    public final <T extends ConfigurationItem> void create(T... entity) {
        validator.validateCis(Arrays.asList(entity));
        repositoryService.create(entity);
    }

    @SafeVarargs
    @Override
    public final <T extends ConfigurationItem> void update(T... entity) {
        validator.validateCis(Arrays.asList(entity));
        repositoryService.update(entity);
    }

    @SafeVarargs
    @Override
    public final <T extends ConfigurationItem> void createOrUpdate(T... entity) {
        validator.validateCis(Arrays.asList(entity));
        repositoryService.createOrUpdate(entity);
    }

    @Override
    public void delete(String... id) {
        repositoryService.delete(id);
    }

    @Override
    public void move(String id, String newId) {
        repositoryService.move(id, newId);
    }

    @Override
    public void rename(String id, String newName) {
        repositoryService.rename(id, newName);
    }

    @Override
    public <T extends ConfigurationItem> List<T> search(Type type) {
        List<T> cis = repositoryService.listEntities(new SearchParameters().setType(type));
        externalPropertiesResolver.resolveExternalProperties(cis);
        return cis;
    }

    @Override
    public <T extends ConfigurationItem> List<T> search(Type type, String parent) {
        List<T> cis = repositoryService.listEntities(new SearchParameters().setType(type).setParent(parent));
        externalPropertiesResolver.resolveExternalProperties(cis);
        return cis;
    }
}
