package com.xebialabs.xlplatform.repository.sql

import jakarta.annotation.PreDestroy
import javax.sql.DataSource
import slick.jdbc._
import slick.util.AsyncExecutor

import scala.concurrent.Await
import scala.concurrent.duration.Duration
import scala.util.{Failure, Success, Try}

sealed trait DatabaseType {
  def profile: JdbcProfile
}

object DatabaseType {

  trait DatabaseProfile extends JdbcProfile {
    override def quoteIdentifier(id: String): String = id
  }

  case object H2 extends DatabaseType {
    val profile = new H2Profile with DatabaseProfile
  }

  case object Derby extends DatabaseType {
    val profile = new DerbyProfile with DatabaseProfile
  }

  case object MySQL extends DatabaseType {
    val profile = new MySQLProfile with DatabaseProfile
  }

  case object Postgres extends DatabaseType {
    val profile = new PostgresProfile with DatabaseProfile
  }

  case object Oracle extends DatabaseType {
    val profile = new OracleProfile with DatabaseProfile {
      override type RowsPerStatement = slick.jdbc.RowsPerStatement
      override def defaultRowsPerStatement: RowsPerStatement = RowsPerStatement.All
    }
  }

  case object DB2 extends DatabaseType {
    val profile = new DB2Profile with DatabaseProfile
  }

  case object SQLServer extends DatabaseType {
    val profile = new SQLServerProfile with DatabaseProfile
  }

}

case class DatabaseConfig(databaseType: DatabaseType,
                          name: String,
                          minThreads: Int = 10,
                          maxThreads: Int = 10,
                          queueSize: Int = 1000,
                          maxConnections: Int = 10,
                          queryTimeout: Duration = Duration.Inf)


object Database {
  var instance: Database = _

  def databaseType: DatabaseType = instance.config.databaseType

  def apply(dataSource: DataSource, config: DatabaseConfig): Database = {
    this.instance = new Database(dataSource, config)
    this.instance
  }
}

class Database(dataSource: DataSource, val config: DatabaseConfig) {
  lazy val database: slick.jdbc.JdbcBackend.Database = createDatabaseFrom(dataSource, config)

  def runAwait[R](action: slick.dbio.DBIOAction[R, slick.dbio.NoStream, scala.Nothing]): R = {
    runAwaitTry(action) match {
      case Failure(cause: Throwable) => throw new RuntimeException("Failed to execute DB operation", cause)
      case Success(result) => result
    }
  }

  def runAwaitTry[R](action: slick.dbio.DBIOAction[R, slick.dbio.NoStream, scala.Nothing]): Try[R] =
    withDatabase { db =>
      Await.result(db.run(action.asTry), config.queryTimeout)
    }

  def withDatabase[TResult](action: slick.jdbc.JdbcBackend.Database => TResult): TResult = action(database)

  private def createDatabaseFrom(dataSource: DataSource, config: DatabaseConfig): slick.jdbc.JdbcBackend.Database = {
    slick.jdbc.JdbcBackend.Database.forDataSource(dataSource,
      Some(config.maxConnections),
      AsyncExecutor(s"executor-${config.name}", config.minThreads, config.maxThreads, config.queueSize, config.maxConnections, registerMbeans = true)
    )
  }

  @PreDestroy
  def close(): Unit = {
    if (database != null) {
      database.close()
    }
  }
}
