package com.xebialabs.xlplatform.scheduler

import akka.actor.ActorRef
import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.CONFIGURATION
import com.xebialabs.deployit.repository.{RepositoryService, SearchParameters}
import com.xebialabs.xlplatform.scheduler.ControlTaskExecutor.Messages.InvokeControlTask
import com.xebialabs.xlplatform.scheduler.Scheduler.Messages.{CancelJob, ScheduleJob}
import com.xebialabs.xlplatform.scheduler.ci.{ControlTaskInvokable, ScheduledJob}
import com.xebialabs.xlplatform.scheduler.spring.ServiceHolder
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe

import scala.collection.convert.wrapAll._
import scala.collection.mutable
import scala.reflect.ClassTag

@DeployitEventListener class EventListener(val scheduler: ActorRef, val controlTaskExecutor: ActorRef, val repositoryService: RepositoryService) extends Logging {

  def this() = this(SchedulerActorSystem.scheduler, SchedulerActorSystem.controlTaskExecutor, ServiceHolder.getRepositoryService)

  implicit val actorSystem = SchedulerActorSystem.actorSystem
  implicit val executionContext = actorSystem.dispatcher

  @Subscribe def receiveSystemStart(event: SystemStartedEvent): Unit = {
    logger.debug("Starting scheduled control task jobs")
    val scheduledJobType = Type.valueOf(classOf[ScheduledJob])
    logger.debug(s"Searching all nodes of type $scheduledJobType")
    val cis = repositoryService.list(new SearchParameters().setType(scheduledJobType).setAncestor(CONFIGURATION.getRootNodeName).setResultsPerPage(-1L))
    logger.debug(s"Found ${cis.size()} jobs matching criteria")
    cis.foreach { cid => startJob(repositoryService.read[ScheduledJob](cid.getId))}
  }

  @Subscribe def receiveCisCreated(event: CisCreatedEvent): Unit = {
    filterCis[ScheduledJob](event).foreach(startJob)
  }

  @Subscribe def receiveCisUpdated(event: CisUpdatedEvent): Unit = {
    filterCis[ScheduledJob](event).foreach(restartJob)
  }

  @Subscribe def receiveCisDeleted(event: CisDeletedEvent): Unit = {
    filterCis[ScheduledJob](event).foreach(stopJob)
  }

  @Subscribe def receiveSystemStop(event: SystemStoppedEvent): Unit = {
    logger.info("Stopping scheduled control task jobs")
    SchedulerActorSystem.shutdownTheSystem()
  }

  private[scheduler] def filterCis[T <: ConfigurationItem](event: CiBaseEvent)(implicit ct: ClassTag[T]): mutable.Buffer[T] = {
    event.getCis.filter(_.getType.isSubTypeOf(Type.valueOf(ct.runtimeClass))).map(_.asInstanceOf[T])
  }

  private[scheduler] def restartJob(job: ScheduledJob) {
    stopJob(job)
    startJob(job)
  }

  private[scheduler] def startJob(job: ScheduledJob) {
    if (job.enabled) {
      val message = job match {
        case x: ControlTaskInvokable => InvokeControlTask(x.getConfigurationItemId(), x.getControlTaskName(), x.getControlTaskParams)
        case _ => throw new IllegalArgumentException(s"Unable to schedule unknown job $job!")
      }
      scheduler ! ScheduleJob(ActorJob(job.getId, controlTaskExecutor, message), job.getTrigger)

    }
  }

  private[scheduler] def stopJob(job: ScheduledJob) {
    scheduler ! CancelJob(job.getId)
  }
}


