package com.xebialabs.deployit.security;

import java.util.List;
import org.springframework.security.core.Authentication;

/**
 * Reads roles from the repository and allows editing. The ordinary {@code getSomething()} methods return Roles <i>without</i> the assigned principals.
 * Only {@link #readRoleAssignments()} will a return a list with Role objects that also contain the principals.
 * <p>
 * To edit the roles, first get a local copy of everything using {@link #readRoleAssignments()}, then write them all back with {@link #writeRoleAssignments(List)}.
 */
public interface RoleService {

    //
    // Simple getters
    //

    /**
     * Returns the global roles from the repository.
     */
    List<Role> getRoles();

    /**
     * Returns the global roles associated with the passed principal.
     */
    List<Role> getRolesFor(String principal);

    /**
     * Returns the global roles associated with the passed authentication.
     */
    List<Role> getRolesFor(Authentication auth);

    /**
     * Returns the local roles present on the passed configuration item.
     */
    List<Role> getRoles(String onConfigurationItem);

    /**
     * Resolves the role name to a role (either global or local).
     */
    Role getRoleForRoleName(String roleName);

    //
    // Edit methods
    //

    /**
     * Returns the global roles with member principals and roles loaded.
     */
    List<Role> readRoleAssignments();

    /**
     * Returns the local roles present on the passed configuration item with member principals and roles loaded.
     */
    List<Role> readRoleAssignments(String onConfigurationItem);

    /**
     * Persists the passed roles as global roles in the system.
     */
    void writeRoleAssignments(List<Role> roles);

    /**
     * Persists the passed roles as local roles specific to the passed configuration item.
     */
    void writeRoleAssignments(String onConfigurationItem, List<Role> roles);

}
