package com.xebialabs.deployit.core;


import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.stream.Collectors;

import static java.lang.String.format;
import static java.lang.reflect.Array.newInstance;

@SuppressWarnings("serial")
public abstract class AbstractStringView<T extends Collection<StringValue>> implements Collection<String>, Serializable {
    protected final T wrapped;

    protected AbstractStringView(T wrapped) {
        this.wrapped = wrapped;
    }

    @Override
    public int size() {
        return wrapped.size();
    }

    @Override
    public boolean isEmpty() {
        return wrapped.isEmpty();
    }

    @Override
    public boolean contains(Object o) {
        return wrapped.contains(stringToStringValue(o));
    }

    @Override
    public Iterator<String> iterator() {
        return wrapped.stream().map(StringValue::toString).iterator();
    }

    @Override
    public Object[] toArray() {
        Object[] objects = new Object[size()];
        Iterator<String> iterator = iterator();
        for (int i = 0; i < objects.length; i++) {
            objects[i] = iterator.next();
        }
        return objects;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <U> U[] toArray(U[] a) {
        int size = size();
        U[] r = a.length >= size ? a :
                (U[]) newInstance(a.getClass().getComponentType(), size);
        Iterator<String> iterator = iterator();
        for (int i = 0; i < r.length; i++) {
            r[i] = (U) iterator.next();
        }
        return r;
    }

    public boolean add(String stringValue) {
        return wrapped.add(new StringValue(stringValue));
    }

    public boolean add(StringValue stringValue) {
        return wrapped.add(stringValue);
    }

    public boolean addEncrypted(String stringValue) {
        return wrapped.add(new EncryptedStringValue(stringValue));
    }

    public abstract AbstractStringView<T> encrypt();

    @Override
    public boolean remove(Object o) {
        return wrapped.remove(stringToStringValue(o));
    }

    @Override
    public boolean containsAll(Collection<?> c) {
        return wrapped.containsAll(transformToStringValue(c));
    }

    @Override
    public boolean addAll(Collection<? extends String> c) {
        boolean changed = false;
        for (String s : c) {
            changed |= add(s);
        }
        return changed;
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        return wrapped.retainAll(transformToStringValue(c));
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        return wrapped.removeAll(transformToStringValue(c));
    }

    protected Collection<StringValue> transformToStringValue(Collection<?> c) {
        return c.stream().map(o -> {
            if (o instanceof String) {
                return new StringValue((String) o);
            } else if (o instanceof StringValue) {
                return (StringValue) o;
            }
            throw new IllegalStateException("Cannot convert input [" + o + "]");

        }).collect(Collectors.toList());
    }

    protected Object stringToStringValue(Object o) {
        if (o instanceof String) {
            return new StringValue((String) o);
        }
        return o;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof AbstractStringView<?>) {
            @SuppressWarnings("unchecked")
            AbstractStringView<Collection<StringValue>> object = (AbstractStringView<Collection<StringValue>>) obj;
            return (wrapped == null && object.wrapped == null) || (wrapped != null && wrapped.equals(object.wrapped));
        }

        return false;
    }

    @Override
    public String toString() {
        return format("%s%s", this.getClass().getSimpleName(), wrapped);
    }

    @Override
    public int hashCode() {
        return wrapped.hashCode();
    }

    @Override
    public void clear() {
        wrapped.clear();
    }

    public T getWrapped() {
        return wrapped;
    }
}
