package com.xebialabs.xlplatform.test.jcr;

import java.io.IOException;
import javax.jcr.Node;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import com.xebialabs.deployit.JcrCredentials;
import com.xebialabs.license.service.LicenseService;
import com.xebialabs.xlplatform.test.RandomPasswordEncrypterProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.jcr.JcrTemplateHolder;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.repository.JcrPathHelper;
import com.xebialabs.deployit.repository.JcrRepositoryService;
import com.xebialabs.deployit.repository.RepositoryMetadataService;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlplatform.repository.JcrRepositoryMetadataService;
import com.xebialabs.xlplatform.repository.XlRepositoryConfig;
import com.xebialabs.xlplatform.test.ci.NoLicenseService;

public class JcrTestHelper {
    private static final Logger logger = LoggerFactory.getLogger(JcrTestHelper.class);

    private JcrTemplate jcrTemplate;
    private RepositoryProvider booter;
    private PasswordEncrypter passwordEncrypter;
    private RepositoryMetadataService repositoryMetadataService;

    public JcrTestHelper() {
    }

    public JcrTestHelper(RepositoryProvider provider, XlRepositoryConfig config) throws IOException, RepositoryException {
        setupRepository(provider, config);
    }

    public void setupRepository(RepositoryProvider provider, XlRepositoryConfig config) throws RepositoryException, IOException {
        this.booter = provider;
        passwordEncrypter = new RandomPasswordEncrypterProvider().randomPassword();
        Repository repo = booter.boot(config);
        jcrTemplate = new JcrTemplate(repo, ((JcrCredentials) config.credentials()).creds());
        repositoryMetadataService = new JcrRepositoryMetadataService(jcrTemplate);
        // Set the template in the holder
        new JcrTemplateHolder(jcrTemplate);
    }

    public void shutdownRepository() {
        booter.shutdown();
    }

    public Repository getRepository() {
        return booter.getRepository();
    }

    public JcrRepositoryService getRepositoryService() {
        return getRepositoryService(new NoLicenseService());
    }

    public JcrRepositoryService getRepositoryService(LicenseService licenseService) {
        JcrRepositoryService jcrRepositoryService = new JcrRepositoryService(jcrTemplate, passwordEncrypter, licenseService);
        jcrRepositoryService.initializeLicenseValidator();
        return jcrRepositoryService;
    }

    public JcrTemplate getJcrTemplate() {
        return jcrTemplate;
    }

    public RepositoryMetadataService getRepositoryMetadataService() {
        return repositoryMetadataService;
    }

    public void cleanupRepository() {
        jcrTemplate.execute(new JcrCallback<Void>() {
            @Override
            public Void doInJcr(Session session) throws RepositoryException {
                JcrTools jcrTools = new JcrTools();
                for (Metadata.ConfigurationItemRoot root : Metadata.ConfigurationItemRoot.values()) {
                    if (root != Metadata.ConfigurationItemRoot.NESTED) {
                        int removed = jcrTools.removeAllChildren(session, JcrPathHelper.getAbsolutePathFromId(root.getRootNodeName()));
                        logger.info("Removed {} children from {}", removed, root);
                    }
                }
                clearRepositoryId(session);
                session.save();
                return null;
            }
        });
    }

    private void clearRepositoryId(final Session session) throws RepositoryException {
        final Node node = session.getNode(JcrConstants.CONFIGURATION_NODE_ID);
        if (node.hasProperty(JcrConstants.REPOSITORY_ID_PROPERTY_NAME)) {
            node.getProperty(JcrConstants.REPOSITORY_ID_PROPERTY_NAME).remove();
        }
        repositoryMetadataService = new JcrRepositoryMetadataService(jcrTemplate);
    }
}
