package com.xebialabs.xlrelease.versioning.ascode.scm.connector

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.scm.connector._
import com.xebialabs.xlrelease.scm.data.VersionInfo
import org.eclipse.jgit.api.PullResult
import org.eclipse.jgit.lib.{Constants, Ref, Repository}
import org.eclipse.jgit.revwalk.RevWalk
import org.eclipse.jgit.treewalk.TreeWalk
import org.eclipse.jgit.treewalk.filter.PathFilter

import java.util.Date
import scala.jdk.CollectionConverters._
import scala.util.{Try, Using}

class AsCodeJGitConnector(config: FolderVersioningSettings) extends JGitConnector {
  private val gitConnection = config.gitConnection

  override val getConfigId: String = getName(config.getFolderId)

  override lazy val getConnectionSettings: GitConnectionSettings = new GitConnectionSettings(
    gitConnection.getUrl,
    config.branch,
    gitConnection.getAuthenticationMethod.name(),
    gitConnection.getUsername,
    PasswordEncrypter.getInstance().ensureDecrypted(gitConnection.getPassword),
    gitConnection.getDomain,
    gitConnection.getProxyHost,
    gitConnection.getProxyPort,
    gitConnection.getProxyUsername,
    PasswordEncrypter.getInstance().ensureDecrypted(gitConnection.getProxyPassword),
    gitConnection.getProxyDomain
  )

  // TODO check if we can get rid of this stuff for versioning
  // For thread-local settings for Http (authentication settings)
  private var gitClient: GitClient = _
  try {
    gitClient = new GitClient(getConnectionSettings)
  } catch {
    case t: Throwable =>
      throw ScmException("Unable to initialize git client with current settings", t)
  }

  def resetLocalRepo(): Unit = {
    val git = getOrInitGit()
    gitHardResetToOrigin(git)
  }

  def pull(): PullResult = {
    val git = getOrInitGit()
    gitHardResetToOrigin(git)
    pullWithRecovery(git)
  }

  def listVersions(): Seq[VersionInfo] = {
    val git = getOrInitGit()
    val repository = git.getRepository
    val refDatabase = repository.getRefDatabase
    val tagRefs = refDatabase.getRefsByPrefix(generatePrefix(config.getEffectiveTagPrefix)).asScala.toSeq

    getCommitInfo(repository, tagRefs)
  }

  def pullAndListVersions(): Seq[VersionInfo] = {
    val git = getOrInitGit()
    gitHardResetToOrigin(git)
    pullWithRecovery(git)
    listVersions()
  }

  def getVersion(tagName: String): VersionInfo = {
    val git = getOrInitGit()
    val repository = git.getRepository
    val tagRef = findTagRef(repository, tagName)
    getCommitInfo(repository, List(tagRef)).head
  }

  def checkout(filePath: String, tagName: String, reset: Boolean = true): Try[ScmBlobs] = Try {
    val git = getOrInitGit()
    if (reset) {
      gitHardResetToOrigin(git)
    }
    val repository = git.getRepository

    val tagRef = findTagRef(repository, tagName)

    Using.resource(new RevWalk(repository)) { revWalk =>
      val commit = revWalk.parseCommit(tagRef.getObjectId)
      val tree = commit.getTree
      Using.resource(new TreeWalk(repository)) { treeWalk =>
        treeWalk.addTree(tree)
        treeWalk.setRecursive(true)
        treeWalk.setFilter(PathFilter.create(filePath))
        if (!treeWalk.next) throw ScmException(s"No definition file found for tag [$tagName] and path [$filePath]")
        val objectId = treeWalk.getObjectId(0)
        val loader = repository.open(objectId)
        val fileContent = loader.getBytes()
        ScmBlobs(Seq(BinaryFile(filePath, () => fileContent)))
      }
    }
  }

  private def getCommitInfo(repository: Repository, tagRefs: Seq[Ref]): Seq[VersionInfo] = {
    Using.resource(new RevWalk(repository)) { revWalk =>
      tagRefs.map { tagRef =>
        val commit = revWalk.parseCommit(tagRef.getObjectId)
        val version = new VersionInfo
        version.name = tagRef.getName.substring(10) // strip the 'refs/tags/'
        version.shortMessage = commit.getShortMessage
        version.fullMessage = commit.getFullMessage
        version.author = commit.getAuthorIdent.getName
        version.commiter = commit.getCommitterIdent.getName
        version.commitHash = commit.getName
        version.commitTime = new Date(commit.getCommitTime * 1000L)
        version
      }
    }
  }

  private def findTagRef(repository: Repository, tagName: String): Ref = {
    val tagRef = repository.findRef(s"${Constants.R_TAGS}$tagName")
    if (tagRef == null) {
      throw ScmException(s"Unable to find tag [$tagName] on branch [${config.branch}]")
    }
    tagRef
  }

  private def generatePrefix(effectiveTagPrefix: String): String = {
    if (effectiveTagPrefix.endsWith("/")) {
      s"${Constants.R_TAGS}${effectiveTagPrefix}"
    } else {
      //ensure prefix ends with '/' to constrain search to exact prefix match
      s"${Constants.R_TAGS}${effectiveTagPrefix}/"
    }
  }
}
