package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.domain.events.{ConfigurationCreatedEvent, ConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.service.CiIdService
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.versioning.ascode.ValidationMessage
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class ConfigurationAsCodeService @Autowired()(ciIdService: CiIdService,
                                              configurationRepository: ConfigurationRepository,
                                              referenceSolver: ReferenceSolver,
                                              eventBus: XLReleaseEventBus) extends Logging {

  def process(context: ImportContext, configuration: Configuration): ImportResult = {
    val metadata = context.metadata
    logger.debug(s"Processing configuration ${configuration.toString} with metadata ${metadata.toString()}")

    find(context, configuration) match {
      case Some(existing) =>
        logger.debug(s"Updating configuration: ${existing.toString}")
        update(context, existing, configuration)
      case None =>
        logger.debug(s"Creating configuration: ${configuration.toString}")
        create(context, configuration)
    }
  }

  private def find(context: ImportContext, configuration: Configuration): Option[Configuration] = {
    val folderId = context.scope.getFolderId
    val title = configuration.getTitle
    val matches = configurationRepository
      .findAllByTypeAndTitle(configuration.getType, title, folderId.orNull, folderId.isDefined).asScala.toList

    if (matches.length > 1) {
      logger.debug(s"There are multiple configurations of type [${configuration.getType}] named [${title}] ${context.scope.description()}, those are: ${matches.toString}")
      throw new AsCodeException(s"Multiple configurations of type [${configuration.getType}] are named [${title}] ${context.scope.description()}. Can not determine which one to update.")
    }

    matches.headOption
  }

  private def populateConfiguration(configuration: Configuration, id: String, references: List[CiReference], folderId: Option[String]): Unit = {
    configuration.setId(id)
    configuration.setFolderId(folderId.orNull)
    referenceSolver.resolveReferences(configuration, references, folderId.orNull)

    CiHelper.fixUpInternalReferences(configuration)
  }

  private def create(context: ImportContext, configuration: Configuration): ImportResult = {
    val id = ciIdService.getUniqueId(Type.valueOf(classOf[Configuration]), CUSTOM_CONFIGURATION_ROOT)
    populateConfiguration(configuration, id, context.references, context.scope.getFolderId)
    val messages = validate(context, configuration)
    val created = configurationRepository.create(configuration, context.scope.getFolderCiUid.orNull)
    ImportResult(
      List(CI.ids.withCreated(created.getId)),
      Seq(() => eventBus.publish(ConfigurationCreatedEvent(created))),
      Map.empty,
      messages.map(msg => ValidationMessage.updateCiId(msg, created.getId())) // need to insert ids
    )
  }

  private def update(context: ImportContext, existing: Configuration, configuration: Configuration): ImportResult = {
    populateConfiguration(configuration, existing.getId, context.references, context.scope.getFolderId)
    val messages = validate(context, configuration)
    val updated = configurationRepository.update(configuration, context.scope.getFolderCiUid.orNull)
    ImportResult(
      List(CI.ids.withUpdated(updated.getId)),
      Seq(() => eventBus.publish(ConfigurationUpdatedEvent(updated))),
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, configuration: Configuration): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(configuration, context.getFolderInfo()).toList
      case None => List.empty
    }
  }
}
