package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.ascode.utils.StaticVariables.XLR_TEMPLATE_KIND
import com.xebialabs.xlrelease.ascode.yaml.model.ImportsSpec
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.ascode.yaml.writer.XLRDefinitionWriter
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings._
import com.xebialabs.xlrelease.scm.connector.{BinaryFile, ScmBlobs, ScmException}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import com.xebialabs.xlrelease.versioning.ascode.scm.YamlUtils
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnector
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.FilePerTypeStrategy.{FILE_PER_CI_TYPE, FILE_PER_CI_TYPE_VALUE}
import com.xebialabs.xltype.serialization.CiReference
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.io.ByteArrayOutputStream
import java.nio.charset.StandardCharsets
import scala.collection.mutable.ListBuffer

object FilePerTypeStrategy {
  val FILE_PER_CI_TYPE = "filePerCIType"
  val FILE_PER_CI_TYPE_VALUE = "One file per type"
}

@Component
class FilePerTypeStrategy @Autowired()(permissions: PermissionChecker,
                                       folderAsCodeService: FolderAsCodeService,
                                       generateService: GenerateService,
                                       definitionParser: XLRDefinitionParser) extends VersioningStyleStrategy {

  override def generateFolder(folderId: String, config: FolderVersioningSettings, filePath: String): GeneratedFolder = {
    try {
      val definitions = generateMultiFileDefinitions(folderId, config)

      val generatedMultiFileContents = definitions.foldLeft(List[GeneratedFolder]())((accumulator, fileDefinitions) => {
        val stream = new ByteArrayOutputStream()
        val generateCtxs = XLRDefinitionWriter().writeYaml(stream, WriterConfig(fileDefinitions._2, writeSecrets = false, writeDefaults = false))
        val secrets = generateCtxs.flatMap(_.secrets).toMap
        accumulator :+ GeneratedFolder(
          ScmBlobs(Seq(BinaryFile(fileDefinitions._1, () => stream.toByteArray))),
          fileDefinitions._2,
          secrets
        )
      })

      val allBinaryFiles = generatedMultiFileContents.foldLeft(Seq[BinaryFile]())((accumulator, generatedFolder) => {
        accumulator ++ generatedFolder.blob.files
      })

      val allsecrets = generatedMultiFileContents.foldLeft(Map[String, String]())((accumulator, generatedFolder) => {
        accumulator ++ generatedFolder.secrets
      })

      val allFileDefinitions = generatedMultiFileContents.foldLeft(List[Definition]())((accumulator, generatedFolder) => {
        accumulator ++ generatedFolder.definitions
      })
      GeneratedFolder(ScmBlobs(allBinaryFiles), allFileDefinitions, allsecrets)

    } catch {
      case e: AsCodeException => throw ScmException(e.getMessage, e.getCause)
    }
  }

  override def fetchChildDefinitionsIfNecesarry(version: String,
                                                config: FolderVersioningSettings,
                                                scmConnector: AsCodeJGitConnector,
                                                definitionsInReleaseFile: List[Definition]): List[Definition] = {
    val filesToBeImported = definitionsInReleaseFile.head.spec.asInstanceOf[ImportsSpec].imports.map(getDefinitionsPath(config, _))
    val importedDefinitions = ListBuffer.empty[Definition]
    if (filesToBeImported.nonEmpty) {
      scmConnector.checkout(filePath = Array(filesToBeImported: _*), version).files.foreach(file => {
        val yamlStr = new String(file.getContent(), StandardCharsets.UTF_8)
        val specs = YamlUtils.preprocessYaml(yamlStr)
        importedDefinitions.addAll(specs.map(definitionParser.parse))
      })
      // Construct a single CIspec for Template Kind specs, only two specs possible here Template and Permissions
      importedDefinitions.groupBy(defition => defition.kind).foldLeft(List[Definition]()) {
        case (accumulator, (kind, definitions)) =>
          if (XLR_TEMPLATE_KIND.equals(kind)) {
            val apiDefinition = definitions.head.apiVersion
            val metaData = definitions.head.metadata
            val ciSpecs = definitions.map(_.spec.asInstanceOf[CiSpec])
            val ciSpec = ciSpecs.foldLeft(CiSpec(List[ConfigurationItem](), List[CiReference]())) { case (accumulator, ciSpec) =>
              CiSpec(accumulator.cis ++ ciSpec.cis, accumulator.references ++ ciSpec.references)
            }
            accumulator :+ Definition(apiDefinition, metaData, kind, ciSpec)
          } else {
            accumulator :+ definitions.head
          }
      }
    } else {
      importedDefinitions.toList
    }
  }

  override def getDefinitionsFilenamesFrom(definition: Definition): List[String] = List(FOLDER_VERSIONING_YAML_FILENAME) ++ definition.spec.asInstanceOf[ImportsSpec].imports

  override def getVersionableFileNames(folderId: String, folderVersioningSettings: FolderVersioningSettings): List[String] = {
    val releaseFileName = getDefinitionsPath(folderVersioningSettings)
    val importSpecDefinition = generateMultiFileDefinitions(folderId, folderVersioningSettings)(releaseFileName).head
    List(FOLDER_VERSIONING_YAML_FILENAME) ++ importSpecDefinition.spec.asInstanceOf[ImportsSpec].imports
  }

  override def name: String = FILE_PER_CI_TYPE

  override def value: String = FILE_PER_CI_TYPE_VALUE

  private def generateMultiFileDefinitions(folderId: String, config: FolderVersioningSettings): scala.collection.mutable.Map[String, List[Definition]] = {
    val map = scala.collection.mutable.Map[String, List[Definition]]()
    val ciConfigsWithFileName = getCiConfigsWithFileName(config)

    validateGeneratePermissions(folderId)
    val scope = FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId)
    val excludeIds = Seq(config.gitConnection.getId)

    ciConfigsWithFileName.filter(!_.cisConfig.isEmpty()).foreach({ ciConfigWithFileName =>
      try {
        map.put(getDefinitionsPath(config, ciConfigWithFileName.fileName),
          generateService.generate(
            GeneratorConfig(None, scope, ciConfigWithFileName.cisConfig, permissions.hasGlobalPermission(ADMIN), excludedEntities = excludeIds)))
      } catch {
        case e: AsCodeException =>
          e.getMessage match {
            case GenerateService.EMPTY_DEFINITION_ERROR =>
            case _ => throw ScmException(e.getMessage, e.getCause)
          }
      }
    })
    map.put(getDefinitionsPath(config), List(generateService.
      generateImportsSpecDefinition(map.keys.map(getfilename).toList).get))
    map
  }

  private def validateGeneratePermissions(folderId: String): Unit = {
    if (!(permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId) || permissions.isCurrentUserAdmin)) {
      throw ScmException(s"Unable to generate configuration: must have Admin or 'Generate Configuration' permission on the folder")
    }
  }

  private def getCiConfigsWithFileName(config: FolderVersioningSettings) = {
    Seq(CiConfigWithFileName(CisConfig(generateConfigurations = config.exportConfiguration), CONNECTIONS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateDashboards = config.exportDashboards), DASHBOARDS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateTemplates = config.exportTemplates), TEMPLATES_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateTriggers = config.exportTriggers), TRIGGERS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateVariables = config.exportVariables), VARIABLES_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateDeliveryPatterns = config.exportPatterns), PATTERNS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateNotificationSettings = config.exportNotifications), NOTIFICATIONS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generatePermissions = config.exportSecurity), PERMISSIONS_VERSIONING_YAML_FILENAME))
  }

  private case class CiConfigWithFileName(cisConfig: CisConfig, fileName: String)


}
