package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.ascode.yaml.parser.XLRDefinitionParser
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings._
import com.xebialabs.xlrelease.scm.connector.ScmException
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.GENERATE_FOLDER_CONFIGURATION
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.FilePerTypeStrategy.{FILE_PER_CI_TYPE, FILE_PER_CI_TYPE_VALUE}
import org.springframework.stereotype.Component

object FilePerTypeStrategy {
  val FILE_PER_CI_TYPE = "filePerCIType"
  val FILE_PER_CI_TYPE_VALUE = "One file per type"
}

/**
 * All entities of a selected type are assumed to be /  will be in one file. I.e. all templates are in one file, all triggers in a separate (but only one) file
 * etc.
 */
@Component
class FilePerTypeStrategy(permissions: PermissionChecker,
                          folderAsCodeService: FolderAsCodeService,
                          generateService: GenerateService,
                          definitionParser: XLRDefinitionParser) extends BaseMultiFileStrategy {

  override def name: String = FILE_PER_CI_TYPE

  override def value: String = FILE_PER_CI_TYPE_VALUE

  def generateMultiFileDefinitions(folderId: String, settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    validateGeneratePermissions(folderId)

    val map = scala.collection.mutable.Map[String, List[Definition]]()
    val configs = getCiConfigsWithFileName(settings)
    val scope = FolderSearch(folderAsCodeService.getFolderPath(folderId), folderId)
    val excludeIds = Seq(settings.gitConnection.getId)

    configs.foreach { config =>
      try {
        val generatorConfig = GeneratorConfig(None, scope, config.cisConfig, permissions.hasGlobalPermission(ADMIN), excludeIds)
        val definitionsPath = getDefinitionsPath(settings.scmPath, config.fileName)

        map.put(definitionsPath, generateService.generate(generatorConfig))
      } catch {
        case e: AsCodeException =>
          e.getMessage match {
            case GenerateService.EMPTY_DEFINITION_ERROR =>
            case _ => throw ScmException(e.getMessage, e.getCause)
          }
      }
    }

    map.put(getDefinitionsPath(settings.scmPath), List(generateService.generateImportsSpecDefinition(map.keys.map(getfilename).toList).get))
    map.toMap
  }

  private def validateGeneratePermissions(folderId: String): Unit = {
    if (!(permissions.hasPermission(GENERATE_FOLDER_CONFIGURATION, folderId) || permissions.isCurrentUserAdmin)) {
      throw ScmException(s"Unable to generate configuration: must have Admin or 'Generate Configuration' permission on the folder")
    }
  }

  private def getCiConfigsWithFileName(config: FolderVersioningSettings) =
    Seq(CiConfigWithFileName(CisConfig(generateConfigurations = config.exportConfiguration), CONNECTIONS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateDashboards = config.exportDashboards), DASHBOARDS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateTemplates = config.exportTemplates), TEMPLATES_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateTriggers = config.exportTriggers), TRIGGERS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateVariables = config.exportVariables), VARIABLES_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateDeliveryPatterns = config.exportPatterns), PATTERNS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generateNotificationSettings = config.exportNotifications), NOTIFICATIONS_VERSIONING_YAML_FILENAME),
      CiConfigWithFileName(CisConfig(generatePermissions = config.exportSecurity), PERMISSIONS_VERSIONING_YAML_FILENAME))
      .filter(!_.cisConfig.isEmpty())

  private case class CiConfigWithFileName(cisConfig: CisConfig, fileName: String)

  override def definitionParser(): XLRDefinitionParser = definitionParser

  override def maxNumOfFilesGenerated: Integer = 9
}
