package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings.FOLDER_VERSIONING_ID_ROOT
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.service.{CiIdService, ConfigurationService}
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils

import scala.jdk.CollectionConverters._

/**
 * Separated from FolderVersioningService in order to avoid circular dependency with as-code where it depends on GenerateService
 */
@Service
class FolderVersioningConfigService(ciIdService: CiIdService,
                                    configurationRepository: ConfigurationRepository,
                                    configurationService: ConfigurationService) {

  private val ciType = Type.valueOf(classOf[FolderVersioningSettings])

  def createOrUpdateSettings(newConfig: FolderVersioningSettings): FolderVersioningSettings = {
    val config = findSettings(newConfig.getFolderId) match {
      case Some(existingConfig) =>
        existingConfig.updateWith(newConfig)
        existingConfig
      case None =>
        newConfig.setId(ciIdService.getUniqueId(ciType, FOLDER_VERSIONING_ID_ROOT))
        newConfig
    }

    configurationService.createOrUpdate(config)
    config
  }

  def deleteSettings(folderId: String): Unit = {
    findSettings(folderId) match {
      case Some(config) => configurationRepository.delete(config.getId)
      case None =>
    }
  }

  def findSettings(folderId: String): Option[FolderVersioningSettings] =
    configurationRepository.
      findAllByTypeAndTitle[FolderVersioningSettings](ciType, null, folderId, folderOnly = true).asScala.headOption

  def updateSettingsDirectly(config: FolderVersioningSettings): FolderVersioningSettings = configurationRepository.update(config)

  def validateConfigData(config: FolderVersioningSettings, folderTitle: String): Unit = {
    checkArgument(StringUtils.hasText(config.scmPath),
      s"Can't create version control configurations for folder $folderTitle, scmPath cannot be blank")
  }
}
