package com.xebialabs.xlrelease.domain

import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service.{ExactSearch, FolderSearch, SearchScope}
import com.xebialabs.xlrelease.repository.Ids

import java.util.{ArrayList => JArrayList, List => JList}
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

object FolderExportConfig {
  def apply(): FolderExportConfig = new FolderExportConfig()

  private def toSearchScope(exportConfig: FolderExportConfig): SearchScope = {
    if (exportConfig.includeEntities.isEmpty) {
      FolderSearch(path = null, folderId = exportConfig.folderId, includeSubFolders = exportConfig.includeSubFolders)
    } else {
      ExactSearch(ids = exportConfig.includeEntities.asScala.toList)
    }
  }

  private def toCisConfig(exportConfig: FolderExportConfig): CisConfig = {
    val outOfScope = false
    CisConfig(
      generatePermissions = exportConfig.includeKinds.permissions,
      generateRoles = outOfScope,
      generateUsers = outOfScope,
      generateEnvironments = outOfScope,
      generateApplications = outOfScope,
      includeSecrets = false,
      generateTemplates = exportConfig.includeKinds.templates,
      generateWorkflows = exportConfig.includeKinds.workflows,
      generateTriggers = exportConfig.includeKinds.triggers,
      generateDeliveryPatterns = exportConfig.includeKinds.deliveryPatterns,
      generateDashboards = exportConfig.includeKinds.dashboards,
      generateConfigurations = exportConfig.includeKinds.configurations,
      generateSettings = outOfScope,
      generateRiskProfiles = outOfScope,
      generateVariables = exportConfig.includeKinds.variables,
      generateNotificationSettings = exportConfig.includeKinds.notifications,
      generateCalendar = outOfScope
    )
  }

  def toGeneratorConfig(exportConfig: FolderExportConfig, isAdminUser: Boolean): GeneratorConfig = {
    val maybeName = Option(exportConfig.nameFilter).filterNot(_.isBlank)
    GeneratorConfig(maybeName, toSearchScope(exportConfig), toCisConfig(exportConfig), isAdminUser, exportConfig.excludeEntities.asScala.toSeq)
  }

  def validateAndNormalize(exportConfig: FolderExportConfig): Unit = {
    if (exportConfig.folderId == null || exportConfig.folderId.isBlank) {
      throw new IllegalArgumentException("folderId is required")
    }
    else if (!Ids.isFolderId(exportConfig.folderId)) {
      throw new IllegalArgumentException(s"${exportConfig.folderId} is not a folder id")
    }

    val hasKinds = exportConfig.includeKinds.hasKinds()

    if (!hasKinds && exportConfig.includeEntities.isEmpty) {
      throw new IllegalArgumentException("You need to specify kinds or entities to export")
    }

    // If entities are specified only those kinds will be included
    // supporting a hybrid of specific items and all of other kinds requires
    // refactoring the generators to support multiple scopes
    if (!exportConfig.includeEntities.isEmpty) {
      val entities = exportConfig.includeEntities.asScala

      if (!entities.forall(Ids.isReleaseId)) {
        throw new IllegalArgumentException("Only template ids can be specified in includeEntities")
      }

      if (!entities.forall(_.startsWith(exportConfig.folderId))) {
        throw new IllegalArgumentException("Only ids in the same folder or subfolders can be specified")
      }

      val entityExists = entities.exists(Ids.isReleaseId)
      exportConfig.includeKinds.templates = entityExists
      exportConfig.includeKinds.workflows = entityExists
    }
  }
}

class FolderExportKinds {
  @BeanProperty
  var configurations: Boolean = false

  @BeanProperty
  var dashboards: Boolean = false

  @BeanProperty
  var deliveryPatterns: Boolean = false

  @BeanProperty
  var notifications: Boolean = false

  @BeanProperty
  var permissions: Boolean = false

  @BeanProperty
  var templates: Boolean = false

  @BeanProperty
  var workflows: Boolean = false

  @BeanProperty
  var triggers: Boolean = false

  @BeanProperty
  var variables: Boolean = false

  def hasKinds(): Boolean = {
    configurations ||
      dashboards ||
      deliveryPatterns ||
      notifications ||
      permissions ||
      templates ||
      workflows ||
      triggers ||
      variables
  }
}

class FolderExportConfig {
  @BeanProperty
  var folderId: String = _

  @BeanProperty
  var nameFilter: String = _

  @BeanProperty
  var includeSubFolders: Boolean = false

  @BeanProperty
  var includeKinds: FolderExportKinds = new FolderExportKinds

  @BeanProperty
  var includeEntities: JList[String] = new JArrayList[String]

  @BeanProperty
  var excludeEntities: JList[String] = new JArrayList[String]
}
