package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.overthere.local.LocalConnection
import com.xebialabs.overthere.{OverthereConnection, OverthereFile}
import grizzled.slf4j.Logging

import java.io.InputStream
import java.nio.file.{Files, Paths}
import java.util.zip.ZipFile
import scala.jdk.CollectionConverters._

class TempFileResolver(in: InputStream) extends Logging with FileResolver {
  private var tempFiles: Map[String, OverthereFile] = Map.empty

  private lazy val connection: OverthereConnection = LocalConnection.getLocalConnection
  private lazy val zipFile: ZipFile = open()

  override def resolve(ref: String): Option[OverthereFile] = {
    getLocalOverthereFile(ref)
  }

  override def close(): Unit = {
    cleanup(connection, zipFile)
  }

  private def open(): ZipFile = {
    logger.debug("Opening TempFileMapper")
    val archiveFile = createTempFile("archive.zip", in)
    val zipFile = new ZipFile(Paths.get(archiveFile.getPath).toFile)
    debugListZipEntries(zipFile)
    zipFile
  }

  private def createTempFile(filename: String, in: InputStream) = {
    val outputFile = connection.getTempFile(filename)
    val outputPath = Paths.get(outputFile.getPath)
    logger.debug(s"Writing temporary file for `$filename` at `${outputFile.toString}`")
    Files.copy(in, outputPath)
    outputFile
  }

  private def debugListZipEntries(zip: ZipFile): Unit = {
    if (logger.isDebugEnabled) {
      logger.debug("Listing ZipEntries")
      zip.entries().asScala.foreach { entry =>
        logger.debug(s"ZipEntry [name: `${entry.getName}`, directory: ${entry.isDirectory}, size: ${entry.getSize} bytes], crc: [${entry.getCrc}]")
      }
    }
  }

  private def getLocalOverthereFile(path: String): Option[OverthereFile] = {
    if (tempFiles.contains(path)) {
      logger.debug(s"Reusing file for `$path`")
      tempFiles.get(path)
    } else {
      Option(zipFile.getEntry(path)).map(entry => {
        if (entry.isDirectory) {
          throw new AsCodeException(s"Not allowed to reference a directory: `$path`. Directories should be zipped.")
        } else {
          val in = zipFile.getInputStream(entry)
          val fileName = Paths.get(path).getFileName.toString
          val outputFile = createTempFile(fileName, in)
          tempFiles = tempFiles + (path -> outputFile)
          outputFile
        }
      })
    }
  }

  private def cleanup(connection: OverthereConnection, zipFile: ZipFile): Unit = {
    logger.debug("Cleaning up")
    zipFile.close()
    connection.close()
  }
}
