package com.xebialabs.xlrelease.versioning.ascode.actors

import com.xebialabs.xlplatform.cluster.ClusterMode.{FULL, HOT_STANDBY, STANDALONE}
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.initializer.ActorInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.versioning.ascode.actors.FolderVersioningActor.FolderVersioningCommand
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.AsCodeJGitConnectorInitializer
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersioningConfigService, FolderVersioningService}
import org.apache.pekko.actor.{Actor, ActorLogging, ActorRef, Props}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Component

import scala.concurrent.{Await, Promise}

@Component
class FolderVersioningActorHolder @Autowired()(xlrConfig: XlrConfig) {
  private val actorPromise = Promise[ActorRef]()

  def actorRef(): ActorRef = {
    Await.result(actorPromise.future, xlrConfig.timeouts.systemInitialization)
  }

  def resolveActorRef(actorRef: ActorRef): Unit = {
    actorPromise.success(actorRef)
  }
}

@Component
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredFolderVersioningActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                                folderVersioningActorHolder: FolderVersioningActorHolder,
                                                                folderVersioningService: FolderVersioningService,
                                                                folderVersioningConfigService: FolderVersioningConfigService,
                                                                connectorInitializer: AsCodeJGitConnectorInitializer)
  extends ActorInitializer {
  override def initialize(): Unit = {
    folderVersioningActorHolder.resolveActorRef(
      systemHolder.actorSystem.actorOf(
        Props(new NonClusteredFolderVersioningActor(folderVersioningService, connectorInitializer, folderVersioningConfigService) with FolderVersioningActorAuthMixin)
      )
    )
  }
}

@Component
@Profile(Array(FULL))
class ClusteredFolderVersioningActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                             folderVersioningActorHolder: FolderVersioningActorHolder,
                                                             folderVersioningService: FolderVersioningService,
                                                             folderVersioningConfigService: FolderVersioningConfigService,
                                                             connectorInitializer: AsCodeJGitConnectorInitializer)
  extends ActorInitializer {
  override def initialize(): Unit = {
    folderVersioningActorHolder.resolveActorRef(
      systemHolder.actorSystem.actorOf(
        Props(new ClusteredFolderVersioningActor(folderVersioningService, connectorInitializer, folderVersioningConfigService) with FolderVersioningActorAuthMixin)
      )
    )
  }
}

trait FolderVersioningActorAuthMixin extends Actor {
  this: ActorLogging =>
  abstract override def receive: Receive = {
    case cmd: FolderVersioningCommand =>
      val maybeAuthentication = cmd.callerContext
      log.debug(s"Setting caller as ${maybeAuthentication.map(_.getName)} for command $cmd")
      val newCtxt = SecurityContextHolder.createEmptyContext()
      newCtxt.setAuthentication(maybeAuthentication.orNull)
      SecurityContextHolder.setContext(newCtxt)
      try {
        super.aroundReceive(super.receive, cmd)
      } finally {
        SecurityContextHolder.clearContext()
      }
    case cmd =>
      super.aroundReceive(super.receive, cmd)
  }
}
