package com.xebialabs.xlrelease.versioning.ascode.repository.sql.persistence

import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.json.JsonUtils
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.versioning.ascode.CiInfo
import com.xebialabs.xlrelease.versioning.ascode.form.{ExternalDependencyReportMessage, ValidationReportMessage}
import com.xebialabs.xlrelease.versioning.ascode.repository.sql.persistence.FolderValidationMessagesSchema.FOLDER_VALIDATION_MESSAGES
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet

@Repository
@IsTransactional
class FolderValidationMessagePersistence @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                                      @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect
                                                     ) extends PersistenceSupport with Logging {

  private val STMT_FIND_BY_ID =
    s"""|SELECT *
        |FROM ${FOLDER_VALIDATION_MESSAGES.TABLE} fvm
        |WHERE fvm.${FOLDER_VALIDATION_MESSAGES.MESSAGE_ID} = :id
     """.stripMargin

  def findById(id: Integer): Option[FolderValidationMessageRow] = {
    sqlQuery(STMT_FIND_BY_ID,
      params("id" -> id),
      folderValidationMessageRowMapper).headOption
  }

  private val STMT_FIND_BY_FOLDER_ID =
    s"""|SELECT *
        |FROM ${FOLDER_VALIDATION_MESSAGES.TABLE} fvm
        |WHERE fvm.${FOLDER_VALIDATION_MESSAGES.FOLDER_ID} = :folderId""".stripMargin

  def findByFolderId(folderId: String): Seq[FolderValidationMessageRow] = {
    sqlQuery(STMT_FIND_BY_FOLDER_ID,
      params("folderId" -> folderId),
      folderValidationMessageRowMapper).toSeq
  }

  private val STMT_INSERT_INTO =
    s"""|INSERT INTO ${FOLDER_VALIDATION_MESSAGES.TABLE}
        |(${FOLDER_VALIDATION_MESSAGES.FOLDER_ID}
        | ,${FOLDER_VALIDATION_MESSAGES.CONTENT})
        |VALUES
        |(
        | :folderId
        | , :content
        |)
     """.stripMargin

  def insert(folderId: String, fvm: ValidationReportMessage): Integer = {
    val json = serializeValidationReportMessage(fvm);
    sqlInsertWithContent(pkName(FOLDER_VALIDATION_MESSAGES.MESSAGE_ID), STMT_INSERT_INTO,
      params("folderId" -> folderId),
      "content" -> json,
      ciUid => ciUid
    )
  }

  private val STMT_UPDATE_MESSAGE =
    s"""
       | UPDATE ${FOLDER_VALIDATION_MESSAGES.TABLE}
       | SET ${FOLDER_VALIDATION_MESSAGES.CONTENT} = :content
       | WHERE
       | ${FOLDER_VALIDATION_MESSAGES.MESSAGE_ID} = :messageId
    """.stripMargin

  def update(id: Integer, fvm: ValidationReportMessage): Unit = {
    val json = serializeValidationReportMessage(fvm);
    sqlExecWithContent(STMT_UPDATE_MESSAGE,
      params("messageId" -> id),
      "content" -> json,
      () => _
    )
  }

  private val STMT_DELETE_FOLDER_VALIDATION_MESSAGE =
    s"""
       |DELETE FROM ${FOLDER_VALIDATION_MESSAGES.TABLE}
       |WHERE ${FOLDER_VALIDATION_MESSAGES.MESSAGE_ID} = :messageId
     """.stripMargin

  def delete(messageId: Integer): Boolean = {
    sqlExec(STMT_DELETE_FOLDER_VALIDATION_MESSAGE, params("messageId" -> messageId), _.execute())
  }

  private def serializeValidationReportMessage(fvm: ValidationReportMessage): String = {
    JsonUtils.objectMapper.writeValueAsString(fvm)
  }

  val folderValidationMessageRowMapper: RowMapper[FolderValidationMessageRow] = (rs: ResultSet, _: Int) => {
    val content = ResourceManagement.using(rs.getBinaryStream(FOLDER_VALIDATION_MESSAGES.CONTENT))(decompress)
    val fvm = JsonUtils.objectMapper.readValue(content, classOf[ExternalDependencyReportMessage])
    val messageId = rs.getInt(FOLDER_VALIDATION_MESSAGES.MESSAGE_ID)
    val folderId = rs.getString(FOLDER_VALIDATION_MESSAGES.FOLDER_ID)

    FolderValidationMessageRow(folderId, fvm.id, fvm.ciType, fvm.ciTypeDisplayName, fvm.folder, fvm.folderId, fvm.title,
      fvm.property, fvm.message, fvm.messageType, messageId, fvm.resolvedStatus,new CiInfo(fvm.id,
                                                                                            fvm.ciType,
                                                                                            fvm.ciTypeDisplayName,
                                                                                            fvm.folder,
                                                                                            fvm.folderId,
                                                                                            fvm.title))
  }
}
