package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.ascode.service.spec.CiSpecInterpreter.ProcessedCi
import com.xebialabs.xlrelease.ascode.utils.{FolderScope, ImportContext}
import com.xebialabs.xlrelease.delivery.events.{DeliveryCreatedFromAsCodeEvent, DeliveryUpdatedFromAsCodeEvent}
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.service.DeliveryPatternService
import com.xebialabs.xlrelease.delivery.util.DeliveryObjectFactory
import com.xebialabs.xlrelease.domain.delivery.Stage.DEFAULT_STAGE_TITLE
import com.xebialabs.xlrelease.domain.delivery._
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.Date
import scala.reflect.ClassTag

@Service
class DeliveryPatternAsCodeService @Autowired()(deliveryPatternService: DeliveryPatternService,
                                                deliveryRepository: DeliveryRepository,
                                                referenceSolver: ReferenceSolver,
                                                ciIdService: CiIdService,
                                                folderAsCodeService: FolderAsCodeService,
                                                eventBus: XLReleaseEventBus) extends Logging {

  protected val factory = new DeliveryObjectFactory(ciIdService)

  def process(context: ImportContext, pattern: Delivery): ImportResult = {
    logger.debug(s"Processing Delivery Pattern: ${pattern.toString} with metadata ${context.metadata.toString}")

    referenceSolver.resolveReferences(pattern, context.references, context.scope.getFolderId.get)
    val processed = find(context, pattern) match {
      case Some(existing) =>
        // For some bizarre reasons patterns enforce name uniqueness globally in the system
        // instead of on a folder level so if we found the pattern in a different folder
        // it should be treated as if attempting to create a new pattern with the same name
        if (!existing.getFolderId.equals(context.scope.getFolderId.get)) {
          val newFolder = context.scope.asInstanceOf[FolderScope].path
          val oldFolder = folderAsCodeService.getFolderPath(existing.getFolderId)
          throw new AsCodeException(s"Can't create pattern ${pattern.getTitle} in folder ${newFolder} because one already exists with that name in ${oldFolder}")
        }
        logger.debug(s"Updating Delivery Pattern: ${existing.toString}")
        val updated = update(context, existing, pattern)
        ProcessedCi(updated, CI.ids.withUpdated(updated.getId), Seq(() => eventBus.publish(DeliveryUpdatedFromAsCodeEvent(updated, context.scmData))))
      case None =>
        logger.debug(s"Creating Delivery Pattern: ${pattern.toString}")
        val created = create(context, pattern, factory.deliveryId(), true)
        ProcessedCi(created, CI.ids.withCreated(created.getId), Seq(() => eventBus.publish(DeliveryCreatedFromAsCodeEvent(created, context.scmData))))
    }

    ImportResult(List(processed.changedIds), processed.postCommitActions)
  }

  private def find(context: ImportContext, pattern: Delivery): Option[Delivery] = {
    deliveryRepository.findPatternByTitle(pattern.getTitle)
  }

  private def update(context: ImportContext, existing: Delivery, pattern: Delivery): Delivery = {
    deliveryRepository.delete(existing.getId)
    create(context, pattern, existing.getId, false)
  }

  private def create(context: ImportContext, pattern: Delivery, id: String, setDefaultStage: Boolean): Delivery = {
    pattern.setId(id)
    pattern.setStatus(DeliveryStatus.TEMPLATE)
    pattern.setFolderId(context.scope.getFolderId.get)

    deliveryPatternService.validatePattern(pattern, false)

    resetPattern(pattern)

    if (pattern.getStages.isEmpty && setDefaultStage) {
      val defaultStage = new Stage(DEFAULT_STAGE_TITLE)
      defaultStage.setId(factory.createUniqueId[Stage](pattern.getId))
      pattern.addStage(defaultStage)
    }

    deliveryPatternService.validateStagesAndTransitionsAndItems(pattern)

    deliveryRepository.create(pattern)
    pattern

  }

  def resetPattern(pattern: Delivery, forceNewIds: Boolean = false): Unit = {
    val resetDate = new Date()

    val oldDeliveryId = pattern.getId
    val newDeliveryId = oldDeliveryId

    pattern.getStages.forEach { stage =>
      val oldStageId = stage.getId
      val newStageId = generateOrUpdateId[Stage](oldStageId, oldDeliveryId, newDeliveryId, forceNewIds)

      stage.setId(newStageId)
      Option(stage.getTransition).foreach(transition => {
        val oldTransitionId = transition.getId
        val newTransitionId = generateOrUpdateId[Transition](transition.getId, oldStageId, newStageId, forceNewIds)

        transition.setId(newTransitionId)
        transition.setStage(stage)
        transition.getAllConditions.forEach { condition =>
          condition.setId(generateOrUpdateId[Condition](condition.getId, oldTransitionId, newTransitionId, forceNewIds))
        }
      })
    }

    pattern.getTrackedItems.forEach { item =>
      // Generate Item Ids
      item.setId(factory.trackedItemId(newDeliveryId))
      item.setCreatedDate(resetDate)
      item.setModifiedDate(resetDate)
    }
  }

  private def generateOrUpdateId[T: ClassTag](existingId: String, oldParentId: String, newParentId: String, forceNewIds: Boolean): String = {
    val typeName = Type.valueOf(implicitly[ClassTag[T]].runtimeClass).getName
    val idNotDefined = existingId == null || existingId.trim.isEmpty || !getName(existingId).startsWith(typeName)
    if (forceNewIds || idNotDefined) {
      factory.createUniqueId[T](newParentId)
    } else {
      existingId.replace(Option(oldParentId).getOrElse(""), Option(newParentId).getOrElse(""))
    }
  }
}
