package com.xebialabs.xlrelease.versioning.ascode.upgrader

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plumbing.CurrentVersion
import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlrelease.builder.FolderBuilder.newFolder
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.Team.{FOLDER_OWNER_TEAMNAME, RELEASE_ADMIN_TEAMNAME, TEMPLATE_OWNER_TEAMNAME}
import com.xebialabs.xlrelease.domain.configuration.HttpConnection
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.folder.Folder.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.connector.Repository
import com.xebialabs.xlrelease.security.XLReleasePermissions.{GLOBAL_AUTHENTICATED_USERS_ROLE, VIEW_FOLDER}
import com.xebialabs.xlrelease.security.sql.snapshots.service.PermissionsSnapshotService
import com.xebialabs.xlrelease.service.{FolderService, SharedConfigurationService, TeamService}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.common.BaseInitializingUpgrade
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.MixedMultiFileStrategy.MULTIPLE_FILES_PER_CI_TYPE
import com.xebialabs.xlrelease.versioning.ascode.scm.{FolderVersionApplyHelper, FolderVersioningService}
import com.xebialabs.xlrelease.versioning.ascode.upgrader.XLRelease241DefaultContentFolderInitializer._
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class XLRelease241DefaultContentFolderInitializer(folderService: FolderService,
                                                  val folderVersioningService: FolderVersioningService,
                                                  sharedConfigurationService: SharedConfigurationService,
                                                  permissionSnapshotService: PermissionsSnapshotService,
                                                  teamService: TeamService,
                                                  xlrConfig: XlrConfig)
  extends BaseInitializingUpgrade with FolderVersionApplyHelper with Logging {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "24.1.0#8")

  override def doUpgrade(): Boolean = {
    logger.info(s"Upgrading ${xlrConfig.features.provision.folders.defaultContentFolder.folderTitle} folder configuration")
    if (xlrConfig.features.provision.folders.enabled) {
      Try {
        val maybeFolder = Try(folderService.findById(DEFAULT_CONTENT_FOLDER_ID)).toOption
        val daiContentFolder: Folder = maybeFolder match {
          case Some(contentFolder) =>
            deleteFolderVersioningSettings(contentFolder)
            contentFolder
          case None =>
            logger.info(s"Default content folder '$DEFAULT_CONTENT_FOLDER_ID' does not exist, going to create it")
            createDefaultContentFolder()
        }
        // get git configuration for root folder
        val gitConfiguration = findGitConfiguration(daiContentFolder.getId)
        val scmBranch = getDefaultContentFolderReleaseBranch()

        // create folder versioning settings on Digital.ai Content folder
        createFolderVersioningSettings(
          daiContentFolder.getId,
          gitConfiguration,
          scmBranch,
          SCM_PATH,
          autoImport = true
        )

        fetchAndApplyLatestVersion(daiContentFolder.getId, daiContentFolder.getTitle)
      } match {
        case Failure(exception) => logger.error("Failure while creating and importing content for the default content folder", exception)
        case Success(_) => logger.debug(s"Finished creation of default content folder")
      }
    } else {
      logger.info(s"Skipped default content folder creation as folder provisioning is disabled")
    }
    true
  }

  private def deleteFolderVersioningSettings(folder: Folder): Unit = {
    val subFolders = folder.getChildren.asScala
    folderVersioningService.deleteSettings(folder.getId)
    subFolders.foreach(deleteFolderVersioningSettings)
  }

  private def createDefaultContentFolder(): Folder = {
    val daiFolderToCreate = newFolder.withId(DEFAULT_CONTENT_FOLDER_ID).withTitle(xlrConfig.features.provision.folders.defaultContentFolder.folderTitle).build
    val daiFolder = folderService.create(ROOT_FOLDER_ID, daiFolderToCreate, createDefaultTeams = false)
    val folderViewerTeam = getFolderViewerTeam()
    teamService.saveTeamsToPlatformWithoutPublishing(daiFolder.getId, folderViewerTeam.asJava, false)

    // Event listeners are not subscribed yet, we need to do snapshot manually
    Try(permissionSnapshotService.makeSnapshot(Option(DEFAULT_CONTENT_FOLDER_ID)))

    daiFolder
  }

  private def findGitConfiguration(parentId: String): Repository = {
    val maybeConfiguration = Try(getGitConfiguration(parentId)).toOption
    maybeConfiguration match {
      case Some(configuration) => configuration
      case None =>
        createGitConfiguration(parentId)
    }
  }

  private def getGitConfiguration(parentId: String): Repository = {
    val configurations = sharedConfigurationService.searchByTypeAndTitle(
      Type.valueOf("git.Repository"),
      xlrConfig.features.provision.folders.defaultContentFolder.repositoryTitle,
      parentId,
      true
    ).asScala

    configurations.headOption
      .getOrElse(throw new NotFoundException("No configuration of type 'git.Repository' found"))
      .asInstanceOf[Repository]
  }

  private def createGitConfiguration(parentId: String): Repository = {
    val gitConfiguration = Type.valueOf("git.Repository").getDescriptor.newInstance[Repository]("")
    gitConfiguration.setTitle(xlrConfig.features.provision.folders.defaultContentFolder.repositoryTitle)
    gitConfiguration.setUrl(xlrConfig.features.provision.folders.defaultContentFolder.repositoryUrl)
    gitConfiguration.setAuthenticationMethod(HttpConnection.AuthenticationMethod.None)
    gitConfiguration.setFolderId(parentId)

    val createdGitConfiguration = sharedConfigurationService.create(gitConfiguration)
    createdGitConfiguration.asInstanceOf[Repository]
  }

  private def createFolderVersioningSettings(parentId: String,
                                             gitRepository: Repository,
                                             scmBranch: String,
                                             scmPath: String,
                                             autoImport: Boolean): FolderVersioningSettings = {
    val versionConfiguration = Type.valueOf(classOf[FolderVersioningSettings]).getDescriptor.newInstance[FolderVersioningSettings]("")
    versionConfiguration.setFolderId(parentId)
    versionConfiguration.branch = scmBranch
    versionConfiguration.gitConnection = gitRepository
    versionConfiguration.scmPath = scmPath
    versionConfiguration.exportConfiguration = false
    versionConfiguration.exportDashboards = false
    versionConfiguration.exportNotifications = false
    versionConfiguration.exportTriggers = false
    versionConfiguration.exportVariables = false
    versionConfiguration.exportSecurity = true
    versionConfiguration.exportApplications = false
    versionConfiguration.exportEnvironments = false
    versionConfiguration.versioningStyle = MULTIPLE_FILES_PER_CI_TYPE
    versionConfiguration.autoImport = autoImport

    folderVersioningService.createOrUpdateSettings(versionConfiguration, remoteValidations = false)
  }

  private def getFolderViewerTeam(): Seq[Team] = {
    val viewer: Team = newTeam(VIEWER_TEAMNAME)
    viewer.getPermissions.add(VIEW_FOLDER.getPermissionName)
    viewer.addRole(GLOBAL_AUTHENTICATED_USERS_ROLE)

    getSystemTeams() :+ viewer
  }

  private def newTeam(teamName: String): Team = {
    val team: Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(null)
    team.setTeamName(teamName)
    team
  }

  private def getSystemTeams(): Seq[Team] = {
    val folderOwner = newTeam(FOLDER_OWNER_TEAMNAME)
    val templateOwner = newTeam(TEMPLATE_OWNER_TEAMNAME)
    val releaseAdmin = newTeam(RELEASE_ADMIN_TEAMNAME)

    Seq(folderOwner, templateOwner, releaseAdmin)
  }

}

object XLRelease241DefaultContentFolderInitializer {
  val DEFAULT_CONTENT_FOLDER_ID = s"${Folder.ROOT_FOLDER_ID}/FolderDefaultReleaseContent"
  val VIEWER_TEAMNAME = "Viewer"
  val SCM_PATH = "DigitalAIOfficial"

  def getDefaultContentFolderReleaseBranch(): String = {
    val releaseVersion = Version.valueOf(XL_RELEASE_COMPONENT, CurrentVersion.get)
    s"release/${releaseVersion.getMajor}.${releaseVersion.getMinor}.x"
  }
}
