package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.xlrelease.api.v1.filter.{ApplicationEnvironmentFilterScope, ApplicationFilters}
import com.xebialabs.xlrelease.ascode.utils.Utils.filterFolderOnGeneratePermissions
import com.xebialabs.xlrelease.domain.environments.Application
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.environments.service.ApplicationService
import com.xebialabs.xlrelease.security.PermissionChecker
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_APPLICATION, EDIT_APP_PIPELINES}

import scala.jdk.CollectionConverters._

@Component
class ApplicationAsCodeGenerator @Autowired()(applicationService: ApplicationService,
                                              permissions: PermissionChecker) extends ApplicationEnvironmentGenerateStrategy[Application] {
  override val generateOrder: Double = 400

  override def isDefinedAt(config: CiGenerateConfig): Boolean = {
    config.generateConfig.generateApplications
  }

  override protected def generateFolder(config: CiGenerateConfig): Map[Option[Folder], List[Application]] = {
    config.folders
      .filter { folder => filterFolderOnGeneratePermissions(config, folder.getId) || permissions.hasPermission(EDIT_APP_PIPELINES, folder.getId) }
      .map { folder =>
        val scope = new ApplicationFilters(ApplicationEnvironmentFilterScope.FOLDER)
        scope.setFolderId(folder.getId)
        val applications = applicationService.searchApplications(scope).asScala.toList
        Some(folder) -> applications
      }
      .toMap
  }

  override protected def generateGlobal(config: CiGenerateConfig): Map[Option[Folder], List[Application]] = {
    if (config.isAdminUser || permissions.hasGlobalPermission(EDIT_APPLICATION)) {
      val applications = applicationService.searchApplications(new ApplicationFilters(ApplicationEnvironmentFilterScope.GLOBAL)).asScala.toList
      Map(None -> applications)
    } else {
      Map.empty
    }
  }

}
