package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.ascode.yaml.model.Role
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.RolesApi
import com.xebialabs.xlrelease.api.v1.views.{PrincipalView, RoleView}
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.service.RolesUtils._
import com.xebialabs.xlrelease.ascode.utils.Utils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.stream.Collectors
import scala.jdk.CollectionConverters._

@Service
class RolesAsCodeService @Autowired()(rolesApi: RolesApi) {
  def generateGlobalRoles(generatorConfig: GeneratorConfig): List[Role] = {
    if (generatorConfig.cisConfig.generateRoles && !generatorConfig.searchScope.isInstanceOf[FolderSearch]) {
      rolesApi
        .getRoles(0, Int.MaxValue)
        .stream()
        .filter { roleView =>
          generatorConfig.name.forall { n => Utils.stringLike(roleView.getName, n) }
        }
        .map[Role](convertRoleViewToRole)
        .collect(Collectors.toList[Role])
        .asScala
        .toList
    } else {
      List.empty
    }
  }

  def handleRole(role: Role): ChangedIds = {
    val roleView = convertRoleToRoleView(role)
    val ids = ROLE.ids
    try {
      val storedRole = rolesApi.getRole(role.name)
      roleView.setPermissions(storedRole.getPermissions)
      rolesApi.update(roleView.getName, roleView)
      ids.withUpdated(roleView.getName)
    } catch {
      case _: NotFoundException =>
        rolesApi.create(roleView.getName, roleView)
        ids.withCreated(roleView.getName)
    }
  }
}

object RolesUtils {
  def convertRoleToRoleView(role: Role): RoleView = {
    val roleView = new RoleView
    roleView.setName(role.name)
    val principals = role.principals.map { principal =>
      val principalView = new PrincipalView
      principalView.setUsername(principal)
      principalView
    }
    roleView.setPrincipals(principals.asJava)
    roleView
  }

  def convertRoleViewToRole(roleView: RoleView): Role = {
    val principals = roleView.getPrincipals.asScala.map(_.getUsername).toSet
    Role(roleView.getName, principals)
  }
}
