package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.xlrelease.api.v1.TriggersApi
import com.xebialabs.xlrelease.ascode.service.referencehandler.generate.FolderFilterGeneratorHandler
import com.xebialabs.xlrelease.ascode.utils.Utils.filterFolderOnGeneratePermissions
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.{ReleaseTrigger, Trigger}
import com.xebialabs.xlrelease.repository.{FolderRepository, Ids, TriggerRepository}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TRIGGER
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService}
import com.xebialabs.xlrelease.triggers.deployment_based.StatusWebhookTrigger
import com.xebialabs.xlrelease.triggers.event_based.EventBasedTrigger
import com.xebialabs.xlrelease.webhooks.mapping.MappedProperty.StringValue
import org.springframework.data.domain.{PageRequest, Sort}
import org.springframework.stereotype.Component

import scala.collection.immutable.ListMap
import scala.jdk.CollectionConverters._

@Component
class TriggerAsCodeGenerator(triggersApi: TriggersApi,
                             folderRepository: FolderRepository,
                             triggerRepository: TriggerRepository,
                             releaseService: ReleaseService,
                             folderService: FolderService,
                             permissions: PermissionChecker) extends GenerateStrategy[Trigger] {
  override val generateOrder: Double = 30

  override def isDefinedAt(config: CiGenerateConfig): Boolean = {
    config.generateConfig.generateTriggers
  }

  override protected def generateFolder(config: CiGenerateConfig): Map[Option[Folder], List[Trigger]] = {
    val triggersByFolders = config.folders
      .filter { folder => filterFolderOnGeneratePermissions(config, folder.getId) || permissions.hasPermission(EDIT_TRIGGER, folder.getId) }
      .map { folder =>
        val totalResultPerPage = 100
        val triggers = CiGenerator.paginate { page =>
          val triggerRepositoryResult = triggerRepository
            .findByFolderId(folder.getId, nestedFolders = false, PageRequest.of(page.toInt, totalResultPerPage, Sort.by("CI_UID").ascending()))
            .getContent
            .asScala.toSeq
          triggerRepositoryResult -> (triggerRepositoryResult.size < totalResultPerPage)
        }

        triggers.foreach {
          case trigger: ReleaseTrigger =>
            trigger.setTemplate(releaseService.findById(trigger.getTemplate).getTitle)
            trigger.setReleaseFolder(Option(trigger.getReleaseFolder).map(resolveFolderPath).orNull)
          case trigger: EventBasedTrigger =>
            trigger.mappedProperties
              .asScala
              .find(_.targetProperty == "templateId")
              .collect { case p: StringValue => releaseService.findById(p.value) }
              .foreach { template =>
                val value = new StringValue("template")
                value.setValue(template.getTitle)
                trigger.mappedProperties.add(value)
              }
            trigger.mappedProperties.removeIf(_.targetProperty == "templateId")

            trigger.mappedProperties
              .asScala
              .find(_.targetProperty == "releaseFolderId")
              .filter {
                case p: StringValue => p.value != null
                case _ => false
              }
              .collect { case p: StringValue => resolveFolderPath(p.value) }
              .foreach { folderPath =>
                val value = new StringValue("releaseFolder")
                value.setValue(FolderFilterGeneratorHandler.referencePath(absolutePath = folderPath, home = Option.empty))
                trigger.mappedProperties.add(value)
              }
            trigger.mappedProperties.removeIf(_.targetProperty == "releaseFolderId")

          case t: StatusWebhookTrigger =>
            t.mappedProperties
              .asScala
              .find(_.targetProperty == "templateId")
              .collect { case p: StringValue => releaseService.findById(p.value) }
              .foreach { template =>
                val value = new StringValue("template")
                value.setValue(template.getTitle)
                t.mappedProperties.add(value)
              }
            t.mappedProperties.removeIf(_.targetProperty == "templateId")

            t.mappedProperties
              .asScala
              .find(_.targetProperty == "releaseFolderId")
              .filter {
                case p: StringValue => p.value != null
                case _ => false
              }
              .collect { case p: StringValue => resolveFolderPath(p.value) }
              .foreach { folderPath =>
                val value = new StringValue("releaseFolder")
                value.setValue(FolderFilterGeneratorHandler.referencePath(absolutePath = folderPath, home = Option.empty))
                t.mappedProperties.add(value)
              }
            t.mappedProperties.removeIf(_.targetProperty == "releaseFolderId")

          case _ => ???
        }
        Some(folder) -> triggers
      }

    ListMap(triggersByFolders: _*)
  }

  private def resolveFolderPath(releaseFolderId: String): String = {
    folderRepository.getPath(releaseFolderId).drop(1).mkString(Ids.SEPARATOR)
  }
}
