package com.xebialabs.xlrelease.versioning.ascode

import com.xebialabs.xlrelease.domain.events.{ConfigurationCreatedEvent, ConfigurationDeletedEvent, ConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.versioning.ascode.FolderVersioningSettings
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener, XLReleaseEventListener}
import com.xebialabs.xlrelease.repository.{FolderRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.versioning.ascode.actors.FolderVersioningActorService
import com.xebialabs.xlrelease.versioning.scheduler.FolderVersioningAutoApplyJobService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

@Service
@EventListener
class FolderVersioningEventHandler(folderVersioningActorService: FolderVersioningActorService,
                                   folderVersioningAutoApplyJobService: FolderVersioningAutoApplyJobService,
                                   folderRepository: FolderRepository)
  extends XLReleaseEventListener with Logging with PersistenceInterceptor[Folder] {

  folderRepository.registerPersistenceInterceptor(this)

  // todo delete local repo when branch or connector change
  @AsyncSubscribe
  def onConfigurationDelete(event: ConfigurationDeletedEvent): Unit = {
    event.conf match {
      case config: FolderVersioningSettings => this.onDelete(config.getFolderId)
      case _ => // nothing
    }
  }

  @AsyncSubscribe
  def onConfigurationCreate(event: ConfigurationCreatedEvent): Unit = {
    event.conf match {
      case config: FolderVersioningSettings =>
        folderVersioningAutoApplyJobService.handleAutoApplyGitVersion(config)
      case _ => // nothing
    }
  }

  @AsyncSubscribe
  def onConfigurationUpdate(event: ConfigurationUpdatedEvent): Unit = {
    event.updated match {
      case config: FolderVersioningSettings =>
        folderVersioningAutoApplyJobService.handleAutoApplyGitVersion(config)
      case _ => // nothing
    }
  }

  override def onDelete(ciId: String): Unit = {
    try {
      folderVersioningActorService.cleanLocalRepo(ciId, clusterWide = true)
    } catch {
      case e: Exception => logger.warn(s"Failed to clean folder Git versioning working directory for folder $ciId", e)
    }

    try {
      folderVersioningAutoApplyJobService.unschedule(ciId)
    } catch {
      case e: Exception => logger.warn(s"Unable to unschedule automatic apply of new folder version changes for folder $ciId", e)
    }
  }
}
