package com.xebialabs.xlrelease.versioning.scheduler

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.versioning.ascode.scm.FolderVersioningService
import com.xebialabs.xlrelease.versioning.ascode.upgrader.XLRelease241DefaultContentFolderInitializer._
import grizzled.slf4j.Logging
import org.springframework.context.event
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

/**
 * This service schedules quartz job to auto import default workflows.
 * We have added upgrader, [[com.xebialabs.xlrelease.versioning.ascode.upgrader.XLRelease241DefaultContentFolderInitializer]] to create default workflow folders.
 * We could not add this scheduling as there was no guarantee on upgrade execution path which resulted in error if Quartz was not initialized.
 * DO NOT expand this for all folders as this could have serious performance problem during system startup.
 * This is fine for workflows folder as we don't expect to have 500+ default workflows folder.
 */
@Service
class ScheduleDefaultContentFolderAutoApplyService(folderService: FolderService,
                                                   folderVersioningService: FolderVersioningService,
                                                   folderVersioningAutoApplyJobService: FolderVersioningAutoApplyJobService,
                                                   xlrConfig: XlrConfig) extends Logging {

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      Try {
        val maybeFolder = Try(folderService.findById(DEFAULT_CONTENT_FOLDER_ID)).toOption
        maybeFolder match {
          case Some(contentFolder) =>
            // unschedule existing jobs
            unscheduleJob(contentFolder)

            val folderVersioningSetting = folderVersioningService.findSettings(contentFolder.getId)
            folderVersioningSetting match {
              case Some(setting) =>
                updateFolderVersioningBranchIfRequired(setting)
              case None => logger.warn(s"Not scheduling quartz jobs as folder '${contentFolder.getTitle}' does not have versioning settings")
            }
          case None =>
            logger.info(s"Not scheduling quartz jobs as default content folder '$DEFAULT_CONTENT_FOLDER_ID' does not exist")
        }
      } match {
        case Failure(ex) => logger.error(s"Failure while scheduling quartz jobs for default content folder auto import", ex)
        case Success(_) => logger.debug(s"Scheduled all required quartz job for default content folder auto import")
      }
    }
  }

  // update folder versioning setting branch to release branch if required
  private def updateFolderVersioningBranchIfRequired(folderVersioningSettings: FolderVersioningSettings): Unit = {
    if (xlrConfig.features.provision.folders.defaultContentFolder.checkScmBranchOnStart) {
      val currentBranch = folderVersioningSettings.getBranch
      val scmBranch = getDefaultContentFolderReleaseBranch()

      if (currentBranch == scmBranch) {
        logger.debug(s"Branch '$scmBranch' is already set on folder '${folderVersioningSettings.getFolderId}'")
        folderVersioningAutoApplyJobService.handleAutoApplyGitVersion(folderVersioningSettings)
      } else {
        // Not calling here handleAutoApplyGitVersion because the same will be triggered as part of onConfigurationUpdate event handler
        logger.info(s"Updating branch from '$currentBranch' to '$scmBranch' on folder '${folderVersioningSettings.getFolderId}'")
        folderVersioningSettings.setBranch(scmBranch)
        folderVersioningService.createOrUpdateSettings(folderVersioningSettings, remoteValidations = false)
      }
    }
  }

  // Unschedule all auto import jobs for the given folder and its sub folders
  // Sub folders are there for legacy reasons as we used to have folder versioning on all sub folders in 23.3.x version
  private def unscheduleJob(folder: Folder): Unit = {
    Try {
      val subFolders = folder.getChildren.asScala
      folderVersioningAutoApplyJobService.unschedule(folder.getId)
      subFolders.foreach(unscheduleJob)
    } match {
      case Failure(exception) => logger.error(s"Failure while unscheduling older auto import jobs for folder '${folder.getTitle}'", exception)
      case Success(_) => logger.debug(s"Unscheduled auto import for folder '${folder.getTitle}'")
    }
  }
}
