package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.ChangedIds
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.APPLICATIONS
import com.xebialabs.deployit.security.PermissionDeniedException
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.utils._
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationMessage
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.plugins.dashboard.api.internal.DashboardResource
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.plugins.dashboard.events.{DashboardCreatedEvent, DashboardUpdatedEvent}
import com.xebialabs.xlrelease.plugins.dashboard.repository.SqlDashboardRepository
import com.xebialabs.xlrelease.plugins.dashboard.service.{DashboardSecurity, DashboardService}
import com.xebialabs.xlrelease.plugins.dashboard.views.{DashboardView, TileView}
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.isReleaseId
import com.xebialabs.xlrelease.service.{CiIdService, ReleaseService}
import com.xebialabs.xlrelease.utils.CiHelper.getNestedCis
import com.xebialabs.xltype.serialization.CiReference
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util
import java.util.{List => JavaList}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Service
class DashboardAsCodeService @Autowired()(dashboardResource: DashboardResource,
                                          dashboardService: DashboardService,
                                          templateService: ReleaseService,
                                          referenceSolver: ReferenceSolver,
                                          folderAsCodeService: FolderAsCodeService,
                                          sqlDashboardRepository: SqlDashboardRepository,
                                          eventBus: XLReleaseEventBus,
                                          ciIdService: CiIdService,
                                          dashboardSecurity: DashboardSecurity
                                         ) {

  def process(importContext: ImportContext, inputDashBoard: Dashboard): ImportResult = {
    importContext.scope match {
      case scope@(GlobalScope | FolderScope(_, _, _, _, _)) =>
        if (inputDashBoard.getParentId != null) {
          checkArgument(!Ids.isFolderId(inputDashBoard.getParentId), "Folders not allowed as parentIds of dashboards in YAML")
          // for legacy release dashboards in separate yaml TODO get rid of this later, it expects a template to already exist in db
          processDashboardLegacy(inputDashBoard, importContext.references, scope.getFolderId, importContext.metadata)
        } else {
          processFolderDashboard(importContext, inputDashBoard)
        }
      case _: TemplateScope => ???
    }
  }

  def setPermissions(inputDashboard: Dashboard) = {
    if (inputDashboard.isGlobalDashboard) {
      dashboardSecurity.validate(inputDashboard)
      dashboardSecurity.savePermissions(inputDashboard)
    }
  }

  def processFolderDashboard(importContext: ImportContext, inputDashboard: Dashboard): ImportResult = {
    val folderId = importContext.scope.getFolderId.orNull
    if (folderId != null) {
      checkCorrectDefinition(inputDashboard)
    }

    setOwnerIfMissing(inputDashboard)

    inputDashboard.setParentId(folderId)

    inputDashboard.getTiles.asScala.foreach { tile =>
      referenceSolver.resolveReferences(tile, importContext.references, folderId)
    }
    inputDashboard.setTiles(inputDashboard.getTiles)
    getFilters(inputDashboard).foreach { filter =>
      referenceSolver.resolveStringReference(filter.withId("-1"), folderId, importContext.scope.getMetadataHome)
    }
    referenceSolver.resolveStringReference(inputDashboard, folderId, importContext.scope.getMetadataHome)
    val dashboards = folderId match {
      case null => dashboardService.search(folderId, inputDashboard.getTitle, false)
        .appended(dashboardService.findDashboardById(Dashboard.HOME_DASHBOARD_ID))
      case _ => dashboardService.search(folderId, inputDashboard.getTitle, false)
    }
    dashboards.filter(_.getTitle == inputDashboard.getTitle) match {
      case Seq() =>
        // create
        inputDashboard.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Dashboard]), APPLICATIONS.getRootNodeName))
        setTileIds(inputDashboard)
        val messages = validate(importContext, inputDashboard)
        val created = sqlDashboardRepository.createDashboard(inputDashboard, false)
        setPermissions(created)
        ImportResult(
          List(CI.ids.withCreated(created.getId)),
          Seq(() => eventBus.publish(DashboardCreatedEvent(created))),
          Map.empty,
          messages
        )
      case Seq(foundDashboard) =>
        // update
        inputDashboard.setId(foundDashboard.getId)
        setTileIds(inputDashboard)
        val messages = validate(importContext, inputDashboard)
        val updatedDashboard = sqlDashboardRepository.updateDashboard(inputDashboard)
        setPermissions(updatedDashboard)
        ImportResult(
          List(CI.ids.withUpdated(updatedDashboard.getId)),
          Seq(() => eventBus.publish(DashboardUpdatedEvent(updatedDashboard))),
          Map.empty,
          messages
        )
      case dashboards if dashboards.length > 1 =>
        throw new AsCodeException(
          s"There are ${dashboards.length} different dashboards with name [${inputDashboard.getTitle}] in the folder. Please enter a unique name.")
    }
  }

  def initializeTemplateDashboard(importContext: ImportContext, inputDashboard: Dashboard): Unit = {
    val templateScope = importContext.scope.asInstanceOf[TemplateScope]
    val folderId = templateScope.getFolderId.orNull
    checkCorrectDefinition(inputDashboard)

    inputDashboard.setParentId(templateScope.id)
    inputDashboard.setId(Utils.buildDashboardIdFromTemplateId(templateScope.id))
    inputDashboard.getTiles.forEach(_.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Tile]), inputDashboard.getId)))

    inputDashboard.getTiles.asScala.foreach { tile =>
      referenceSolver.resolveReferences(tile, importContext.references, folderId)
    }
    inputDashboard.setTiles(inputDashboard.getTiles)
    getFilters(inputDashboard).foreach { filter =>
      referenceSolver.resolveStringReference(filter.withId("-1"), folderId, importContext.scope.getMetadataHome)
    }
    referenceSolver.resolveStringReference(inputDashboard, folderId, importContext.scope.getMetadataHome)
  }

  private def validate(context: ImportContext, dashboard: Dashboard): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(dashboard, context.getFolderInfo()).toList
      case None => List.empty
    }
  }

  private def setTileIds(dashboard: Dashboard): Unit = {
    dashboard.getTiles
      .asScala
      .foreach(_.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Tile]), dashboard.getId)))
  }

  private def processDashboardLegacy(inputDashboard: Dashboard, references: List[CiReference], folderId: Option[String], metadata: Map[String, String]): ImportResult = {
    val tiles = inputDashboard.getTiles
    val dashboardParent = Option(inputDashboard.getParentId)
    val realFolderId = dashboardParent match {
      case Some(_) => processContainerId(inputDashboard, dashboardParent, folderId, metadata)
      case None => folderId
    }
    inputDashboard.setTiles(new util.ArrayList())
    // if parent exists, check if dashboard in template, else create/update independent dashboard in folder
    val result = dashboardParent.map { id =>
      checkCorrectDefinition(inputDashboard)
      // this will update dashboard also (without tiles)
      findDashboardInTemplate(realFolderId.getOrElse(Ids.SEPARATOR), id)
    }.getOrElse {
      createOrUpdateDashboardInFolder(realFolderId, inputDashboard)
    }

    val createdDashboard = dashboardService.findDashboardById(result.dashboardView.getId())

    inputDashboard.setId(createdDashboard.getId)
    inputDashboard.setParentId(createdDashboard.getParentId)
    inputDashboard.setTitle(createdDashboard.getTitle)
    inputDashboard.setTiles(tiles)
    inputDashboard.setProperty[Int]("columns", inputDashboard.getColumns)
    inputDashboard.setProperty[Int]("rows", inputDashboard.getRows)
    inputDashboard.getTiles.asScala.foreach { tile =>
      referenceSolver.resolveReferences(tile, references, realFolderId.orNull)
    }

    val home = metadata.get(MetadataFields.HOMEFOLDER.toString)
    referenceSolver.resolveStringReference(inputDashboard, realFolderId.getOrElse(""), home)
    getFilters(inputDashboard).foreach { filter =>
      referenceSolver.resolveStringReference(filter.withId("-1"), realFolderId.orNull, home)
    }
    dashboardService.updateDashboard(inputDashboard)

    ImportResult(List(result.action))
  }

  private def getFilters(ci: ConfigurationItem): Seq[ReportFilter] = getNestedCis(ci).asScala.flatMap {
    case filters: JavaList[ReportFilter @unchecked] => filters.asScala
    case filter: ReportFilter => Seq(filter)
    case _ => Seq.empty
  }.toSeq

  private def createOrUpdateDashboardInFolder(containerId: Option[String], dashboard: Dashboard): DashboardHelper = {
    val container = containerId.map { container =>
      if (Ids.isFolderId(container)) container else null
    }.orNull

    if (container != null) {
      checkCorrectDefinition(dashboard)
    }

    dashboard.setParentId(container)
    dashboardResource.search(container, dashboard.getTitle).asScala.filter(_.title == dashboard.getTitle).toList match {
      case Nil =>
        val dashboardView = DashboardView(dashboard)
        val createdDashboard = dashboardResource.createDashboard(dashboardView)
        DashboardHelper(createdDashboard, CI.ids.withCreated(createdDashboard.getId()))
      case dashboardView :: Nil =>
        val inputDashboardView = DashboardView(dashboard)
        val updatedDashboard = dashboardResource.updateDashboard(dashboardView.getId(), inputDashboardView)
        DashboardHelper(updatedDashboard, CI.ids.withUpdated(updatedDashboard.getId()))
      case dashboards =>
        throw new AsCodeException(s"There are ${dashboards.size} different dashboards with name [${dashboards.head.getTitle()}] in the folder. Please enter a unique name.")
    }
  }

  private def checkCorrectDefinition(dashboard: Dashboard): Unit = {
    assertIsEmpty(dashboard, "roleEditors")
    assertIsEmpty(dashboard, "roleViewers")
  }

  private def assertIsEmpty(dashboard: Dashboard, field: String): Unit = {
    dashboard.getProperty[JavaList[String]](field).asScala.toList match {
      case Nil =>
      case _ => throw new AsCodeException(s"The field $field from dashboard [${dashboard.getTitle}] should be empty.")
    }
  }

  private def findDashboardInTemplate(folderId: String, parentTemplate: String): DashboardHelper = {
    val templateIds = if (isReleaseId(parentTemplate) && templateService.exists(parentTemplate)) {
      List(parentTemplate)
    } else {
      templateService.findTemplatesByTitle(folderId, parentTemplate, 0, 2, 1).asScala.map(_.getId).toList
    }

    templateIds match {
      case templateId :: Nil =>
        // even if the dashboard wasn't inserted by the template it will be found because dashboards always exist on a template
        val foundDashboard = dashboardResource.getDashboard(Utils.buildDashboardIdFromTemplateId(templateId), refresh = true)
        foundDashboard.setTiles(List.empty[TileView].asJava)
        val updatedDashboard = dashboardResource.updateDashboard(foundDashboard.getId(), foundDashboard)
        DashboardHelper(updatedDashboard, CI.ids.withUpdated(updatedDashboard.getId()))
      case Nil => throw new AsCodeException(s"Template with name [$parentTemplate] not found.")
      case templates =>
        throw new AsCodeException(s"There are ${templateIds.size} different templates with name [$parentTemplate] in the folder. Please enter a unique name.")
    }
  }

  private def processContainerId(inputDashboard: Dashboard, parentDashboard: Option[String], folderId: Option[String], metadata: Map[String, String]): Option[String] = {
    val folderToCreateDashboard = Try(folderAsCodeService.searchParentFolder(Option(inputDashboard.getTitle).getOrElse(""), metadata)) match {
      case Failure(e: PermissionDeniedException) => throw e
      case Success(value) => value
      case _ => None
    }

    folderToCreateDashboard.map(_.getId).orElse(folderId)
  }

  private def setOwnerIfMissing(dashboard: Dashboard): Unit = {
    if (!dashboard.hasOwner) {
      dashboard.setOwner(getAuthenticatedUserName)
    }
  }

  private case class DashboardHelper(dashboardView: DashboardView, action: ChangedIds)
}
