package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.{enforcePermissionsOnGenerate, filterFolderOnGeneratePermissions}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind}
import com.xebialabs.xlrelease.repository.{Ids, Page}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_TEMPLATE
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService}
import com.xebialabs.xlrelease.views.TemplateFilters

import java.util.Optional
import scala.collection.immutable.ListMap
import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait BaseTemplateAsCodeGenerator extends GenerateStrategy[Release] {
  def permissions: PermissionChecker
  def folderService: FolderService
  def releaseService: ReleaseService

  override val generateOrder: Double = 25
  val TOTAL_RESULT_PER_PAGE = 100L

  override protected def generateByIds(templateIds: List[String], config: CiGenerateConfig): Map[Option[Folder], List[Release]] = {
    val templatesByFolders = templateIds.flatMap { templateId =>
      getTemplate(templateId).map { template =>
        val folder = config.folders.find(_.getId == template.findFolderId)
        folder match {
          case Some(folder) =>
            if (!config.folderGeneratePermissions.getOrElse(folder.getId, false)) {
              permissions.checkView(template)
            }
          case _ => permissions.checkView(template)
        }
        folder -> template
      }
    }.groupMap(_._1)(_._2)
    templatesByFolders
  }

  private def checkForDuplicateTitlesInScope(templates: List[Release], knownTitles: mutable.Set[String], folder: Option[Folder]): Unit = {
    templates.foreach(template => {
      val title = template.getTitle
      if (!knownTitles.add(title)) {
        val templateScope = folder match {
          case Some(folder) => s"template in folder ${folder.getTitle}"
          case None => "global template"
        }
        throw new ItemConflictException(s"Found a duplicate. There is more than one $templateScope with title $title, please use an unique title")
      }
    })
  }

  private def folderId(folder: Option[Folder]): String = {
    folder match {
      case Some(folder) => folder.getId
      case None => Ids.ROOT_FOLDER_ID
    }
  }

  private def generateTemplates(config: CiGenerateConfig, folder: Option[Folder]): (Option[Folder], List[Release]) = {
    val titlesInFolder = mutable.Set[String]()
    (folder, CiGenerator.paginate { page =>
      var templates = getTemplates(
        folderId(folder),
        Page.parse(Optional.ofNullable(page), Optional.ofNullable(TOTAL_RESULT_PER_PAGE), Optional.ofNullable(1)),
        enforcePermissionsOnGenerate(config, folderId(folder))
      )
      val totalRetrieved = templates.size
      templates = config.ciTitle match {
        case Some(title) => templates.filter { template => Utils.stringLike(template.getTitle, title) }
        case None => templates
      }
      if (config.checkForDuplicates) {
        checkForDuplicateTitlesInScope(templates, titlesInFolder, folder)
      }
      templates -> (totalRetrieved < TOTAL_RESULT_PER_PAGE)
    })
  }

  override protected def generateGlobal(config: CiGenerateConfig): Map[Option[Folder], List[Release]] = {
    ListMap(generateTemplates(config, None))
  }

  override protected def generateFolder(config: CiGenerateConfig): Map[Option[Folder], List[Release]] = {
    val templatesByFolders = config.folders
      .filter { folder => filterFolderOnGeneratePermissions(config, folder.getId) || permissions.hasPermission(VIEW_TEMPLATE, folder.getId) }
      .map { folder => generateTemplates(config, Some(folder)) }
    ListMap(templatesByFolders: _*)
  }

  override protected def filter(results: Map[Option[Folder], List[Release]], ciTitle: String): Map[Option[Folder], List[Release]] = {
    // Titles were filtered when they were retrieved, no further filtering is necessary
    results
  }

  protected def getSupportedReleaseKind(): ReleaseKind

  private def getTemplates(folderId: String, page: Page, enforcePermission: Boolean): List[Release] = {
    folderService.searchTemplates(new TemplateFilters(folderId, getSupportedReleaseKind()) , page, enforcePermission).asScala.toList
  }

  private def getTemplate(templateId: String): Option[Release] = {
    val releaseKind = releaseService.getReleaseKind(templateId)
    if (releaseKind == getSupportedReleaseKind()) {
      val releaseTemplate = releaseService.findById(templateId)
      Option(releaseTemplate)
    } else {
      None
    }
  }
}
