package com.xebialabs.xlrelease.versioning.ascode.scm

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil.{FOLDER_VERSIONING_YAML_FILENAME, getDefinitionsPath}
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.scm.connector.{BinaryFile, ScmBlobs}
import com.xebialabs.xlrelease.versioning.ascode.scm.FolderVersioningService.wrapExceptions
import com.xebialabs.xlrelease.versioning.ascode.scm.connector.{AsCodeJGitConnector, AsCodeJGitConnectorInitializer}
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.VersioningStrategyResolver
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import scala.util.{Failure, Success, Try, Using}

@Component
class FolderVersioningPreviewService @Autowired()(connectorInitializer: AsCodeJGitConnectorInitializer,
                                                  folderVersioningConfigService: FolderVersioningConfigService,
                                                  versioningStyleResolver: VersioningStrategyResolver,
                                                  importsCrawler: ImportCrawler) {

  def generatePreview(folderId: String, version: Option[String], fileName: String = FOLDER_VERSIONING_YAML_FILENAME): String = wrapExceptions {
    val config = getSettings(folderId)
    val finalFileName = if (fileName.isEmpty) FOLDER_VERSIONING_YAML_FILENAME else fileName
    val filesToCheckout = getFilesForCheckout(finalFileName, config)
    val blobs: BinaryFile = version match {
      case Some(ver) =>
        generateVersionPreview(finalFileName, config, filesToCheckout, ver)
      case None =>
        generateUnversionedPreview(folderId, finalFileName, config)
    }
    new String(blobs.getContent(), StandardCharsets.UTF_8)
  }

  def generatePreview(folderId: String, version: Option[String], filenames: List[String]): Map[String, String] = wrapExceptions {
    val config = getSettings(folderId)
    val filesToCheckout = getFilesForCheckout(filenames, config)

    version match {
      case Some(ver) =>
        Using.resource(connectorInitializer.init(config)) { scmConnector =>
          Try(scmConnector.checkout(filesToCheckout, ver, reset = false)) match {
            case Success(scmBlob) =>
              extractFileContent(filesToCheckout, scmBlob, config.getScmPath)

            case Failure(ex: ScmException) if ex.getMessage.contains("No definition file found for tag") =>
              throw new NotFoundException(ex.getMessage)

            case Failure(ex) =>
              throw ex
          }
        }
      case None =>
        val scmBlob = versioningStyleResolver.resolve(config.getVersioningStyle).generateFolder(folderId, config).blob
        extractFileContent(filesToCheckout, scmBlob, config.getScmPath)
    }
  }

  private def generateVersionPreview(fileName: String, config: FolderVersioningSettings, filesToCheckout: Array[String], ver: String) = {
    val filenameWithPath = getDefinitionsPath(config.getScmPath, fileName)
    Using.resource(connectorInitializer.init(config)) { scmConnector =>
      Try(scmConnector.checkout(filesToCheckout, ver, reset = false)) match {
        case Success(checkedOut) =>
          if (!fileName.equals(FOLDER_VERSIONING_YAML_FILENAME)) {
            verifyRequestedFile(fileName, config, scmConnector, ver)
          }
          getRequestedFile(checkedOut, filenameWithPath)
        case Failure(ex: ScmException) if ex.getMessage.contains("No definition file found for tag") => throw new NotFoundException(ex.getMessage)
        case Failure(ex) => throw ex
      }
    }
  }

  private def generateUnversionedPreview(folderId: String, fileName: String, settings: FolderVersioningSettings) = {

    // TODO optimize - no need to create an entire folder to generate preview
    val allFiles = versioningStyleResolver.resolve(settings.getVersioningStyle).generateFolder(folderId, settings).blob.filesToAdd
    val requestedFile = allFiles.filter(file => file.absolutePath.substring(settings.getScmPath.length + 1).equals(fileName))

    if (requestedFile.isEmpty) {
      throw new NotFoundException(s"Invalid file '$fileName' requested.")
    } else {
      requestedFile.head
    }
  }

  private def getFilesForCheckout(fileName: String, config: FolderVersioningSettings): Array[String] = {
    val scmPath = config.getScmPath
    if (fileName.equals(FOLDER_VERSIONING_YAML_FILENAME)) {
      Array(getDefinitionsPath(scmPath, fileName))
    } else {
      Array(getDefinitionsPath(scmPath, fileName), getDefinitionsPath(scmPath, FOLDER_VERSIONING_YAML_FILENAME))
    }
  }

  private def getFilesForCheckout(fileNames: List[String], config: FolderVersioningSettings): Array[String] = {
    fileNames.flatMap(entry => Seq(getDefinitionsPath(config.getScmPath, entry))).toArray
  }

  private def extractFileContent(filesToCheckout: Array[String], scmBlob: ScmBlobs, scmPath: String) = {
    filesToCheckout.map { fileName =>
      val content = getRequestedFile(scmBlob, fileName)
      val key = fileName.stripPrefix(scmPath + "/")
      val value = new String(content.getContent(), StandardCharsets.UTF_8)
      key -> value
    }.toMap
  }

  def verifyRequestedFile(fileName: String, config: FolderVersioningSettings, scm: AsCodeJGitConnector, version: String): Unit = {
    val imports = importsCrawler.findAllImports(config.getScmPath)(scm, version)
    if (!imports.contains(fileName)) {
      throw ScmException(s"The requested file '$fileName' is not amongst the imports. Available imports are: ${imports.mkString(", ")}.")
    }
  }

  def getRequestedFile(checkedOut: ScmBlobs, fileName: String): BinaryFile = checkedOut.filesToAdd.find(_.absolutePath.equals(fileName)).get

  def getSettings(folderId: String): FolderVersioningSettings = wrapExceptions {
    folderVersioningConfigService.findSettings(folderId).getOrElse(throw ScmException("Folder Git versioning configuration is not defined for this folder."))
  }

}
