package com.xebialabs.xlrelease.versioning.ascode.scm.strategy

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.service.GenerateService.GeneratorConfig
import com.xebialabs.xlrelease.ascode.service.{FolderAsCodeService, FolderSearch, GenerateService}
import com.xebialabs.xlrelease.domain.utils.ScmException
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettingsUtil.getDefinitionsPath
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.versioning.ascode.scm.strategy.SingleFilePerFolderStrategy.{SINGLE_FILE, SINGLE_FILE_VALUE}

object SingleFilePerFolderStrategy {
  val SINGLE_FILE = "singleFilePerFolder"
  val SINGLE_FILE_VALUE = "Multiple files - one for each folder"
}

/**
 * Releasefile.yaml per folder. If there are subfolders, each will have exactly one Releasefile.yaml.
 */
//@Component // hidden
class SingleFilePerFolderStrategy(permissionsChecker: PermissionChecker,
                                  generateService: GenerateService,
                                  folderService: FolderService,
                                  folderAsCodeService: FolderAsCodeService) extends BaseVersioningStrategy {

  override def name: String = SINGLE_FILE

  override def value: String = SINGLE_FILE_VALUE

  override def order: Integer = 20

  override def performGenerate(scope: FolderSearch, rootPath: String, settings: FolderVersioningSettings): Map[String, List[Definition]] = {
    val ciConfigs = settings.toCisConfig
    val excludeIds = Seq(settings.getGitConnection.getId)

    try {
      val generatorConfig = GeneratorConfig(None, scope, ciConfigs, permissionsChecker.hasGlobalPermission(ADMIN), excludeIds)
      val definitions = generateService.generate(generatorConfig)

      Map(getDefinitionsPath(scope.path) -> definitions.map(_.adjustMetadata(rootPath)))
    } catch {
      case e: AsCodeException =>
        e.getMessage match {
          case GenerateService.EMPTY_DEFINITION_ERROR => Map(getDefinitionsPath(settings.getScmPath) -> List.empty)
          case _ => throw ScmException(e.getMessage, e.getCause)
        }
    }
  }

  override def getFolderService: FolderService = folderService

  override def getFolderAsCodeService: FolderAsCodeService = folderAsCodeService

  override def getPermissions: PermissionChecker = permissionsChecker

  override def isMultiFile: Boolean = false
}

