package com.xebialabs.xlrelease.ascode.rest

import com.xebialabs.ascode.annotation.ExceptionToAsCodeResponse
import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model.Definition
import com.xebialabs.ascode.yaml.writer.DefinitionWriter.WriterConfig
import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.core.rest.resteasy.Workdir
import com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean.ONLY_ON_EXCEPTION
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.ascode.service.GenerateService.{CisConfig, GeneratorConfig}
import com.xebialabs.xlrelease.ascode.service._
import com.xebialabs.xlrelease.ascode.service.previewhandler.PreviewService
import com.xebialabs.xlrelease.ascode.utils.StaticVariables
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.SEPARATOR
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.FolderService
import com.xebialabs.xlrelease.versioning.ascode.scm.{DefinitionsGenerator, FolderVersioningService}
import grizzled.slf4j.Logging
import jakarta.ws.rs._
import jakarta.ws.rs.core.{MediaType, Response}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.io.InputStream
import scala.beans.BeanProperty

@Controller
@ExceptionToAsCodeResponse
@Path("/devops-as-code")
class XLRAsCodeResource @Autowired()(creationService: CreationService,
                                     generateService: GenerateService,
                                     previewService: PreviewService,
                                     permissionChecker: PermissionChecker,
                                     folderService: FolderService,
                                     folderVersioningService: FolderVersioningService,
                                     definitionsGenerator: DefinitionsGenerator) extends Logging {
  @POST
  @Produces(Array(MediaType.APPLICATION_JSON))
  @Path("/apply")
  def createPipeline(definition: Definition, scmData: SCMTraceabilityData): Response = {
    try {
      val response = creationService.processBlueprint(definition, scmData)
      Response.ok(response).build()
    } catch {
      case e: Throwable =>
        logger.error(e.getMessage, e)
        throw e
    }
  }

  @POST
  @Path("/apply")
  @Consumes(Array("application/zip"))
  @Produces(Array(MediaType.APPLICATION_JSON))
  @Workdir(prefix = StaticVariables.AS_CODE_GENERATE_PREFIX, clean = ONLY_ON_EXCEPTION)
  def interpretZip(in: InputStream, scmData: SCMTraceabilityData): Response = {
    val response = creationService.processArchive(in, scmData)
    Response.ok(response).build()
  }


  //noinspection ScalaStyle
  @GET
  @Produces(Array("application/zip", "text/vnd.yaml"))
  @Workdir(prefix = StaticVariables.AS_CODE_GENERATE_PREFIX, clean = ONLY_ON_EXCEPTION)
  @Path("/generate")
  def generateCi(@QueryParam("path") path: String,
                 @QueryParam("ciName") ciName: String,
                 @QueryParam("ciIds") ciIds: Array[String],
                 @DefaultValue("false") @QueryParam("defaults") defaults: Boolean,
                 @DefaultValue("false") @QueryParam("permissions") permissions: Boolean,
                 @DefaultValue("false") @QueryParam("users") users: Boolean,
                 @DefaultValue("false") @QueryParam("roles") roles: Boolean,
                 @DefaultValue("false") @QueryParam("environments") environments: Boolean,
                 @DefaultValue("false") @QueryParam("applications") applications: Boolean,
                 @DefaultValue("false") @QueryParam("secrets") includeSecrets: Boolean,
                 @DefaultValue("false") @QueryParam("templates") templates: Boolean,
                 @DefaultValue("false") @QueryParam("workflows") workflows: Boolean,
                 @DefaultValue("false") @QueryParam("triggers") triggers: Boolean,
                 @DefaultValue("false") @QueryParam("dashboards") dashboards: Boolean,
                 @DefaultValue("false") @QueryParam("configurations") configurations: Boolean,
                 @DefaultValue("false") @QueryParam("settings") settings: Boolean,
                 @DefaultValue("false") @QueryParam("riskProfiles") riskProfiles: Boolean,
                 @DefaultValue("false") @QueryParam("deliveryPatterns") deliveryPatterns: Boolean,
                 @DefaultValue("false") @QueryParam("variables") variables: Boolean,
                 @DefaultValue("false") @QueryParam("notifications") notificationSettings: Boolean,
                 @DefaultValue("false") @QueryParam("calendar") calendar: Boolean,
                 @DefaultValue("true") @QueryParam("includeSubFolders") includeSubFolders: Boolean,
                 @DefaultValue("false") @QueryParam("folderVersioning") folderVersioning: Boolean): Response = {

    // Only templates support generate by id
    if (!ciIds.forall(Ids.isReleaseId)) {
      throw new AsCodeException("Only template ids can be specified")
    }

    val searchScope = if (!ciIds.isEmpty) {
      ExactSearch(ciIds.toList)
    } else if (Strings.isNotBlank(path)) {
      if (path == SEPARATOR) {
        GlobalSearch
      } else {
        FolderSearch(path, null, includeSubFolders)
      }
    } else {
      AllSearch
    }

    val cisConfig = normalizeCisConfig(searchScope,
      CisConfig(
        generatePermissions = permissions,
        generateRoles = roles,
        generateUsers = users,
        generateEnvironments = environments,
        generateApplications = applications,
        includeSecrets = includeSecrets,
        generateTemplates = templates,
        generateWorkflows = workflows,
        generateTriggers = triggers,
        generateDeliveryPatterns = deliveryPatterns,
        generateDashboards = dashboards,
        generateConfigurations = configurations,
        generateSettings = settings,
        generateRiskProfiles = riskProfiles,
        generateVariables = variables,
        generateNotificationSettings = notificationSettings,
        generateCalendar = calendar,
        generateFolderVersioning = folderVersioning
      )
    )

    try {
      val generatorConfig = GeneratorConfig(Option(ciName).filter(_.nonEmpty), searchScope, cisConfig, permissionChecker.hasGlobalPermission(ADMIN))
      val definitions = generateService.generate(generatorConfig)
      Response.ok(WriterConfig(definitions, includeSecrets, writeDefaults = defaults)).build()
    } catch {
      case e: Throwable =>
        logger.error(e.getMessage, e)
        throw e
    }
  }

  @POST
  @Path("/preview")
  @Workdir(prefix = StaticVariables.AS_CODE_GENERATE_PREFIX, clean = ONLY_ON_EXCEPTION)
  @Produces(Array(MediaType.APPLICATION_JSON))
  def preview(definition: Definition): Response = {
    val response = previewService.preview(definition)
    Response.ok(response).build()
  }

  @GET
  @Produces(Array(MediaType.APPLICATION_JSON))
  @Path("/duplicates/{folderId:.*Folder[^/-]*}")
  def checkDuplicateTitles(@PathParam("folderId") folderId: String): Response = {
    val config = folderVersioningService.getSettings(folderId)
    val definitions = definitionsGenerator.generateWithDuplicates(folderId, config)
    val mappedDefinitionsByFolderPath = mapDefinitions(definitions)

    val duplicateValuesGroupedByKindAndPath: List[DuplicateValues] =
      mappedDefinitionsByFolderPath.toList
        .sortBy(_._1)
        .flatMap { case (folderPath, kindMap) =>
          kindMap.toList
            .sortBy(_._1)
            .map { case (kind, duplicates) =>
              DuplicateValues(
                DuplicateKey(kind, folderPath),
                duplicates
              )
            }
        }

    Response.ok(duplicateValuesGroupedByKindAndPath).build()
  }

  private def getCiKind(ci: ConfigurationItem): String = {
    val id = ci.getId

    def releaseKindValue(): String = {
      ci.getProperty("kind")
        .asInstanceOf[ReleaseKind]
        .value() match {
        case v if v.contains(StaticVariables.XLR_WORKFLOW_KIND.toLowerCase) =>
          StaticVariables.XLR_WORKFLOWS_KIND
        case _ =>
          StaticVariables.XLR_TEMPLATE_KIND
      }
    }

    if (id.contains(StaticVariables.XLR_DASHBOARD_KIND)) {
      StaticVariables.XLR_DASHBOARDS_KIND
    } else if (id.contains(StaticVariables.XLR_RELEASE_KIND)) {
      releaseKindValue()
    } else if (id.contains(StaticVariables.XLR_CONFIGURATION_KIND)) {
      StaticVariables.XLR_CONNECTIONS_KIND
    } else if (id.contains(StaticVariables.XLR_TRIGGER_KIND)) {
      StaticVariables.XLR_TRIGGERS_KIND
    } else {
      ""
    }
  }

  private def getCiUrl(kind: String, folderId: String): Option[String] = {
    if (kind.contains(StaticVariables.XLR_DASHBOARD_KIND)) {
      Some(s"#/folders/$folderId/${kind.toLowerCase()}?has_filter")
    } else if (kind.contains(StaticVariables.XLR_TEMPLATE_KIND)) {
      Some(s"#/folders/$folderId/${kind.toLowerCase()}?has_filter")
    } else if (kind.contains(StaticVariables.XLR_CONNECTIONS_KIND) || kind.contains(StaticVariables.XLR_CONFIGURATION_KIND)) {
      Some(s"#/folders/$folderId/${StaticVariables.XLR_CONFIGURATION_KIND.toLowerCase()}")
    } else if (kind.contains(StaticVariables.XLR_TRIGGER_KIND)) {
      Some(s"#/folders/$folderId/${kind.toLowerCase()}?has_filter")
    } else if (kind.contains(StaticVariables.XLR_WORKFLOW_KIND)) {
      Some(s"#/folders/$folderId/${kind.toLowerCase()}/${StaticVariables.XLR_TEMPLATE_KIND.toLowerCase()}?has_filter")
    } else {
      None
    }
  }

  private def mapDefinitions(definitions: Set[(String, ConfigurationItem)]): Map[String, Map[String, List[DuplicateValue]]] = {
    definitions.map {
        case (folderPath, ci) =>
          val folder = folderService.findByPath(folderPath, folderPath.split(Ids.SEPARATOR).length - 1)
          val folderId = folder.getId
          val title = ci.getProperty("title").asInstanceOf[String]
          val kind = getCiKind(ci)
          val url = getCiUrl(kind, folderId)
          DuplicateValue(folderId, folderPath, title, kind, url.getOrElse(""))
      }.toList
      .groupBy(_.folderPath)
      .map { case (folderPath, items) =>
        val byKind: Map[String, List[DuplicateValue]] =
          items.groupBy(_.kind)
        folderPath -> byKind
      }
  }

  private def normalizeFolderScope(scope: SearchScope, clientValue: Boolean, folderScopeDefault: Boolean): Boolean = {
    scope match {
      case AllSearch => clientValue
      case GlobalSearch => false
      case _: FolderSearch => clientValue | folderScopeDefault
      case _: ExactSearch => false
    }
  }

  private def normalizeGlobalScope(scope: SearchScope, clientValue: Boolean): Boolean = {
    scope match {
      case AllSearch => clientValue
      case GlobalSearch => clientValue
      case _: FolderSearch => false
      case _: ExactSearch => false
    }
  }

  private def normalizeAnyScope(scope: SearchScope, clientValue: Boolean, folderScopeDefault: Boolean): Boolean = {
    scope match {
      case AllSearch => clientValue
      case GlobalSearch => clientValue
      case _: FolderSearch => clientValue | folderScopeDefault
      case _: ExactSearch => false
    }
  }

  private def validateScopeAndInput(scope: SearchScope, clientValues: CisConfig): Unit = {
    scope match {
      case GlobalSearch if !clientValues.hasGlobalTypes() && clientValues.hasFolderTypes() =>
        throw new AsCodeException("Could not generate a definition for path '/' when only folder-scoped types are requested")
      case FolderSearch(path, _, _) if clientValues.hasGlobalTypes() && !clientValues.hasFolderTypes() =>
        throw new AsCodeException(s"Could not generate a definition for folder '$path' when only global-scoped types are requested")
      case AllSearch if clientValues.isEmpty() =>
        throw new AsCodeException("Could not generate a definition without a path or some types specified")
      case _ =>
    }
  }

  private def normalizeCisConfig(scope: SearchScope, clientValues: CisConfig): CisConfig = {
    validateScopeAndInput(scope, clientValues)

    // When folder scope is needed but all folder types excluded it should default to all folder types
    val needAllFolderTypes = scope match {
      case _: FolderSearch => !clientValues.hasFolderTypes()
      case _ => false
    }

    val needTemplates = scope.isInstanceOf[ExactSearch]

    // Normalize so all values are only enabled when they are relevant to the scope being generated
    CisConfig(
      generatePermissions = normalizeAnyScope(scope, clientValues.generatePermissions, needAllFolderTypes),
      generateRoles = normalizeGlobalScope(scope, clientValues.generateRoles),
      generateUsers = normalizeGlobalScope(scope, clientValues.generateUsers),
      generateEnvironments = normalizeAnyScope(scope, clientValues.generateEnvironments, needAllFolderTypes),
      generateApplications = normalizeAnyScope(scope, clientValues.generateApplications, needAllFolderTypes),
      includeSecrets = clientValues.includeSecrets,
      generateTemplates = needTemplates || normalizeAnyScope(scope, clientValues.generateTemplates, needAllFolderTypes),
      generateWorkflows = needTemplates || normalizeAnyScope(scope, clientValues.generateWorkflows, needAllFolderTypes),
      generateTriggers = normalizeFolderScope(scope, clientValues.generateTriggers, needAllFolderTypes),
      generateDeliveryPatterns = normalizeFolderScope(scope, clientValues.generateDeliveryPatterns, needAllFolderTypes),
      generateDashboards = normalizeAnyScope(scope, clientValues.generateDashboards, needAllFolderTypes),
      generateConfigurations = normalizeAnyScope(scope, clientValues.generateConfigurations, needAllFolderTypes),
      generateSettings = normalizeGlobalScope(scope, clientValues.generateSettings),
      generateRiskProfiles = normalizeGlobalScope(scope, clientValues.generateRiskProfiles),
      generateVariables = normalizeAnyScope(scope, clientValues.generateVariables, needAllFolderTypes),
      generateNotificationSettings = normalizeAnyScope(scope, clientValues.generateNotificationSettings, needAllFolderTypes),
      generateCalendar = normalizeGlobalScope(scope, clientValues.generateCalendar),
      generateFolderVersioning = normalizeFolderScope(scope, clientValues.generateFolderVersioning, needAllFolderTypes)
    )
  }

  private case class DuplicateKey(kind: String, folderPath: String)

  private case class DuplicateValue(folderId: String, folderPath: String, title: String, kind: String, url: String)

  private case class DuplicateValues(@BeanProperty var key: DuplicateKey,
                                     @BeanProperty var duplicates: List[DuplicateValue])
}
