package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.ApplicationApi
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.environments.Application
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationMessage
import com.xebialabs.xlrelease.environments.repository.{ApplicationRepository, EnvironmentRepository}
import com.xebialabs.xltype.serialization.CiReference
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class ApplicationAsCodeService(applicationApi: ApplicationApi,
                               applicationRepository: ApplicationRepository,
                               environmentRepository: EnvironmentRepository) {

  def process(context: ImportContext, application: Application): ImportResult = {
    applicationRepository.findApplicationByTitleAndFolder(application.getTitle, context.scope.getFolderId) match {
      case Some(existing) =>
        update(context, application, existing)
      case None =>
        create(context, application)
    }
  }

  private def create(context: ImportContext, application: Application) = {
    application.setFolderId(context.scope.getFolderId.orNull)
    resolveReferences(context.references, context.scope.getFolderId)
    val messages = validate(context, application)
    val created = applicationApi.create(application)
    ImportResult(List(CI.ids.withCreated(created.getId)),
      Seq.empty,
      Map.empty,
      messages
    )
  }

  private def update(context: ImportContext, application: Application, existing: Application) = {
    application.setId(existing.getId)
    application.setFolderId(context.scope.getFolderId.orNull)
    resolveReferences(context.references, context.scope.getFolderId)
    if (existing.getApplicationSource != null) {
      Option(application.getApplicationSource).foreach(_.setId(existing.getApplicationSource.getId))
    }
    val messages = validate(context, application)
    val updated = applicationApi.update(application)
    ImportResult(List(CI.ids.withUpdated(updated.getId)),
      Seq.empty,
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, application: Application): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(application, context.getFolderInfo()).toList
      case None => List.empty
    }
  }

  private def resolveReferences(references: List[CiReference], folderId: Option[String]): Unit = {
    references.foreach(reference => {
      reference.getProperty.getName match {
        case "environments" =>
          val environments = reference.getIds.asScala.map(ref => {
            try {
              environmentRepository.findEnvironmentByTitleAndFolder(ref, folderId)
                .getOrElse(environmentRepository.findEnvironmentByTitle(ref))
            } catch {
              case e: NotFoundException =>
                if (folderId.isDefined) {
                  environmentRepository.findInFolderByCorrelationId(folderId.get, ref).getOrElse(throw e)
                } else {
                  throw e
                }
            }
          }).toList.asJava
          reference.getCi.setProperty("environments", environments)
      }
    })
  }

}
