import React, { useMemo } from 'react';
import refractor from 'refractor';
import { Diff, Hunk, markEdits, parseDiff, tokenize, TokenNode } from 'react-diff-view';
import { DefaultRenderToken } from 'react-diff-view/types/context';
import 'react-diff-view/style/index.css';
import './react-diff-view.less';

interface ReactDiffViewProps {
    diffString: string;
    isInline?: boolean;
    language?: 'yaml' | 'javascript';
}

const renderToken = (tokenNode: TokenNode, defaultRender: DefaultRenderToken, index: number) => {
    if (tokenNode.type === 'space') {
        return (
            <span className="space" key={index}>
                {tokenNode.children?.map((token, i) => renderToken(token, defaultRender, i))}
            </span>
        );
    }
    return defaultRender(tokenNode, index);
};

export const ReactDiffView = ({ diffString, isInline, language }: ReactDiffViewProps) => {
    const [{ hunks, type }] = parseDiff(diffString, { nearbySequences: 'zip' });
    const tokens = useMemo(
        () => tokenize(hunks, { enhancers: [markEdits(hunks, { type: 'block' })], highlight: true, refractor, language: language || '' }),
        [hunks],
    );

    return (
        <Diff
            className={language ? `language-${language}` : undefined}
            diffType={type}
            hunks={hunks}
            renderToken={renderToken}
            tokens={tokens}
            viewType={isInline ? 'unified' : 'split'}
        >
            {(diffHunks) => (
                <>
                    {diffHunks.map((hunk) => (
                        <Hunk hunk={hunk} key={hunk.content} />
                    ))}
                </>
            )}
        </Diff>
    );
};
