package com.xebialabs.xlrelease.versioning.ascode

import com.xebialabs.ascode.yaml.model.{CiSpec, Definition}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.ascode.utils.TemplateUtils
import com.xebialabs.xlrelease.ascode.yaml.parser.ValueTagPreprocessingConstructor.VALUE_TAG
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.environments.EnvironmentStage
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.variables.{FolderVariables, Variable}
import com.xebialabs.xlrelease.domain.versioning.ascode._
import com.xebialabs.xlrelease.domain.versioning.ascode.settings.FolderVersioningSettings
import com.xebialabs.xlrelease.domain.versioning.ascode.validation._
import com.xebialabs.xlrelease.plugins.dashboard.domain.{Dashboard, Tile}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.service.{FolderService, ReleaseService}
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.versioning.ascode.ExportValidator.CiDependency
import grizzled.slf4j.Logging
import org.springframework.util.StringUtils

import java.util
import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters.RichOptional
import scala.util.Try

object ExportValidator {
  // we might need to support string dependencies later
  case class CiDependency(pd: PropertyDescriptor, references: Seq[ConfigurationItem])

  def validate(definitions: Seq[Definition],
               rootFolder: FolderInfo,
               folderService: FolderService,
               releaseService: ReleaseService,
               fromApply: Boolean,
               config: FolderVersioningSettings): Seq[ValidationMessage] = {
    new ExportValidator(definitions, rootFolder, folderService, releaseService, fromApply, config).validate()
  }
}

class ExportValidator(definitions: Seq[Definition],
                      rootFolder: FolderInfo,
                      folderService: FolderService,
                      releaseService: ReleaseService,
                      fromApply: Boolean,
                      config: FolderVersioningSettings) extends Logging {
  private val allCisInDefinitions = definitions.collect {
    case Definition(_, _, _, CiSpec(cis, _, _)) =>
      CiHelper.getNestedCis(cis.asJava).asScala
  }.flatten.toSet

  def validate(): Seq[ValidationMessage] = {
    definitions.foldLeft(Seq.empty[ValidationMessage]) {
      case (warnings, definition) => warnings ++ validateDefinition(definition, rootFolder)
    }
  }

  private def validateDefinition(definition: Definition, rootFolder: FolderInfo): Seq[ValidationMessage] = {
    definition.spec match {
      case spec: CiSpec => validateCis(spec.cis, rootFolder)
      case _ => Seq.empty[ValidationMessage]
    }
  }

  protected def validateCis(cis: List[ConfigurationItem], parentFolder: FolderInfo): Seq[ValidationMessage] = {
    cis.foldLeft(Seq.empty[ValidationMessage]) {
      case (warnings, ci) => warnings ++ validateCi(ci, parentFolder)
    }
  }

  def validateCi(ci: ConfigurationItem, parentFolder: FolderInfo, releaseFolderOutOfScopeExists: Boolean = true): Seq[ValidationMessage] = {
    ci match {
      case template: Release => validateTemplate(template, parentFolder)
      case folderVariables: FolderVariables => validateVariables(folderVariables.getVariables.asScala.toList, parentFolder)
      case folder: Folder =>
        val nestedCis = folder.getChildren.asInstanceOf[java.util.Set[ConfigurationItem]].asScala.toList
        val relPath = if (parentFolder.getRelativePath.isEmpty) folder.getTitle else s"${parentFolder.getRelativePath}/${folder.getTitle}"
        val absPath = s"${parentFolder.getAbsolutePath}/${folder.getTitle}"
        validateCis(nestedCis, new FolderInfo(folder.getId, relPath, absPath))
      case dashboard: Dashboard => validateDashboard(dashboard, parentFolder)
      case _ =>
        val info = CiInfo(ci, parentFolder)
        validateCiProperties(ci, info, releaseFolderOutOfScopeExists)
    }
  }

  protected def validateCiProperties(topLevelCi: ConfigurationItem, ciInfo: CiInfo, releaseFolderIsOutOfTriggerScope: Boolean = true): Seq[ValidationMessage] = {
    val nestedCisWithDeps = getExternalReferencesByDescriptorsByCis(topLevelCi)
    validateCiProperties(nestedCisWithDeps, ciInfo, releaseFolderIsOutOfTriggerScope)
  }

  protected def validateCiProperties(nestedCisWithDeps: Iterable[(ConfigurationItem, List[CiDependency])], ciInfo: CiInfo, releaseFolderIsOutOfTriggerScope: Boolean): Seq[ValidationMessage] = {
    val (cis, dependencies) = nestedCisWithDeps.unzip
    val passValidations = validatePasswords(cis, ciInfo)
    val depsValidations = dependencies.flatMap(_.flatMap(checkDeps(_, ciInfo)))
    val crtValidation = if (ciInfo.ciType.equals("xlrelease.CreateReleaseTask")) {
      validateCRT(cis, ciInfo)
    } else {
      List.empty
    }
    val triggerValidation = validateTriggerTemplateReference(cis.head, ciInfo, releaseFolderIsOutOfTriggerScope)
    passValidations ++ depsValidations ++ crtValidation ++ triggerValidation
  }

  protected def validatePasswords(cis: Iterable[ConfigurationItem], ciInfo: CiInfo): List[ValidationMessage] = {
    cis.flatMap { nestedCi =>
      val properties = nestedCi.getType.getDescriptor.getPropertyDescriptors.asScala
      properties.flatMap(validatePasswords(nestedCi, _, ciInfo))
    }.toList
  }

  // scalastyle:off cyclomatic.complexity
  protected def validatePasswords(ci: ConfigurationItem, pd: PropertyDescriptor, ciInfo: CiInfo): List[ValidationMessage] = {
    val validationMessages = ListBuffer.empty[ValidationMessage]
    val value = pd.get(ci)
    if (pd.isPassword && value != null) {
      pd.getKind match {
        case PropertyKind.STRING =>
          val usesPlainPassword = containsPassword(pd, value.asInstanceOf[String])
          if (usesPlainPassword) {
            validationMessages.addOne(new HardcodedPassword(ciInfo, pd.getFqn))
          }
        case PropertyKind.SET_OF_STRING | PropertyKind.LIST_OF_STRING =>
          val usesPlainPassword = value.asInstanceOf[java.util.Collection[String]].asScala.exists { v =>
            containsPassword(pd, v)
          }
          if (usesPlainPassword) {
            validationMessages.addOne(new HardcodedPassword(ciInfo, pd.getFqn))
          }
        case PropertyKind.MAP_STRING_STRING =>
          val usesPlainPassword = value.asInstanceOf[java.util.Map[String, String]].asScala.exists { case (_, v) =>
            containsPassword(pd, v)
          }
          if (usesPlainPassword) {
            validationMessages.addOne(new HardcodedPassword(ciInfo, pd.getFqn))
          }
        case _ =>
      }
    }

    validationMessages.toList
  }

  private def checkDeps(ciDependency: CiDependency, ciInfo: CiInfo) = {
    val validationMessages = ListBuffer.empty[ValidationMessage]

    ciDependency.references.foreach { ref =>
      //Using exists instead of contains,Seems like the check is done on stale hashcode or something
      if (!this.allCisInDefinitions.exists(cis => cis.equals(ref))) {
        ref match {
          // don't care about the default risk profile
          case rp: RiskProfile if rp.getId == RiskProfile.DEFAULT_RISK_PROFILE_ID => ()
          case _ =>
            // not sure if all of these types are actually relevant
            val refFolderId = ref match {
              case r: Release => r.findFolderId()
              case c: BaseConfiguration => c.getFolderId
              case t: Trigger => t.getFolderId
              case d: Delivery => d.getFolderId
              case d: Dashboard => d.getParentId
              case f: Folder => f.getId
              case e: EnvironmentStage => ciInfo.folderId
              case _ => null

            }
            // todo check if folder paths are actually needed for ref
            val refFolderInfo = new FolderInfo(refFolderId, null, null)
            val refCiInfo = CiInfo(ref, refFolderInfo)
            validationMessages.addOne(new NonVersionedRef(ciInfo, ciDependency.pd.getFqn, refCiInfo))
        }
      }
    }

    validationMessages.toList
  }

  private def validateCRT(cis: Iterable[ConfigurationItem], ciInfo: CiInfo): List[ValidationMessage] = {
    val validationMessages = ListBuffer.empty[ValidationMessage]
    val createReleaseTask = cis.head.asInstanceOf[CreateReleaseTask]
    val descriptors = cis.head.getType.getDescriptor.getPropertyDescriptors.asScala

    val propertyTemplate = descriptors.find(_.getName.equals("templateId")).get
    val propertyFolder = descriptors.find(_.getName.equals("folderId")).get

    addFolderValidation(createReleaseTask, ciInfo, validationMessages, propertyFolder)
    addTemplateValidation(createReleaseTask, fromApply, ciInfo, validationMessages, propertyTemplate)
    validationMessages.toList
  }

  private def getOptionalFolderById(id: String): Option[Folder] = {
    Try { folderService.findByPath(id, id.split(Ids.SEPARATOR).length - 1) }.toOption
  }

  private def addFolderValidation(createReleaseTask: CreateReleaseTask, ciInfo: CiInfo,
                                  validationMessages: ListBuffer[ValidationMessage], propertyFolder: PropertyDescriptor): Unit = {
    Option(createReleaseTask.getFolderId)
      .filterNot(_.startsWith(Ids.LOCAL_FOLDER))
      .foreach { folderId =>
        val refCiInfo = createRefFolder(folderId)
        validationMessages += new NonVersionedRef(ciInfo, propertyFolder.getFqn, refCiInfo.orNull)
      }
  }

  private def createRefFolder(folderId: String): Option[CiInfo] = {
    getOptionalFolderById(folderId).map { folder =>
      CiInfo(folder.getId, "xlrelease.Folder", "Folder", folderId, folder.getId, folder.getTitle)
    }
  }

  private def addTemplateValidation(createReleaseTask: CreateReleaseTask, fromApply: Boolean, ciInfo: CiInfo,
                                    validationMessages: ListBuffer[ValidationMessage], propertyTemplate: PropertyDescriptor): Unit = {
    def addValidation(ref: Option[CiInfo]): Unit = {
      validationMessages += new NonVersionedRef(ciInfo, propertyTemplate.getFqn, ref.orNull)
    }

    def validateLocalTemplate(): Unit = {
      if (fromApply) {
        val releaseId = TemplateUtils.extractReleaseId(createReleaseTask.getId)
        if (!releaseService.exists(releaseId)) {
          addValidation(None)
        }
      }
    }

    def validateExternalTemplate(templateId: String): Unit = {
      val templatePath = TemplateUtils.getTemplatePathByTemplateId(templateId)
      val optionalFolder = getOptionalFolderById(templatePath)

      optionalFolder match {
        case Some(folder) =>
          val title = TemplateUtils.getTemplateTitleByTemplateId(templateId)
          val releaseId = folder.getId.split(Ids.SEPARATOR).last
          val optionalRelease = releaseService.returnReleaseByFolderIdAndReleaseTitle(releaseId, title).toScala

          optionalRelease match {
            case Some(release) =>
              val refCiInfo = CiInfo(release.getId, "xlrelease.Release", "Release", templatePath, folder.getId, release.getTitle)
              addValidation(Some(refCiInfo))
            case None if fromApply =>
              addValidation(None)
            case _ => // no validation needed
          }
        case None if fromApply =>
          addValidation(None)
        case _ => // no validation needed
      }
    }

    Option(createReleaseTask.getTemplateId).foreach { templateId =>
      if (templateId.startsWith(Ids.LOCAL_FOLDER)) {
        validateLocalTemplate()
      } else {
        validateExternalTemplate(templateId)
      }
    }
  }

  private def containsPassword(pd: PropertyDescriptor, value: String): Boolean = {
    !(value.isEmpty ||
      VariableHelper.containsVariables(value) ||
      Option(pd.getDefaultValue).map(_.asInstanceOf[String]).contains(value))
  }

  private def validateTemplate(template: Release, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    val info = CiInfo(template, parentFolder)
    // Pull the phases out of the template so we only check the template properties first
    val phases = template.getPhases
    template.setPhases(List.empty[Phase].asJava)

    // Pull the variables out of the template so we can validate those separately
    val variables = template.getVariables
    template.setVariables(List.empty[Variable].asJava)

    val templateWarnings = phases.asScala.foldLeft(validateCiProperties(template, info)) {
      case (phaseWarnings, phase) => {
        val phaseInfo = CiInfo(phase, parentFolder, Some(info))
        // Validate the tasks in the phase as separate CIs
        phaseWarnings ++ phase.getTasks.asScala.foldLeft(Seq.empty[ValidationMessage]) {
          case (taskWarnings, task) => {
            val taskInfo = CiInfo(task, parentFolder, Some(phaseInfo))
            taskWarnings ++ validateTask(task, taskInfo, parentFolder)
          }
        }
      }
    }

    val variableWarnings = validateVariables(variables.asScala.toList, parentFolder, Some(info))

    // Restore the phases in the template
    template.setPhases(phases)
    template.setVariables(variables)
    templateWarnings ++ variableWarnings
  }

  private def validateTask(task: Task, info: CiInfo, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    task match {
      case group: TaskGroup =>
        // Pull the tasks out of the group so we only check the group properties first
        val tasks = group.getTasks
        group.setTasks(new util.ArrayList[Task])

        val groupWarnings = tasks.asScala.foldLeft(validateCiProperties(group, info)) {
          // Validate the subtasks in the group as separate CIs
          case (taskWarnings, subtask) => {
            val subtaskInfo = CiInfo(subtask, parentFolder, Some(info))
            taskWarnings ++ validateTask(subtask, subtaskInfo, parentFolder)
          }
        }

        // Restore the tasks in the group, sequential groups include validation on addTasks so add by index
        tasks.asScala.zipWithIndex.foreach {
          case (task, i) => group.addTask(task, i)
        }
        groupWarnings
      case _ => validateCiProperties(task, info)
    }
  }

  private def validateVariables(variables: List[Variable], parentFolder: FolderInfo, templateInfo: Option[CiInfo] = None): Seq[ValidationMessage] = {
    variables.foldLeft(Seq.empty[ValidationMessage]) {
      case (warnings, variable) => {
        val info = CiInfo(variable, parentFolder, templateInfo)
        warnings ++ validateCiProperties(variable, info)
      }
    }
  }

  private def validateDashboard(dashboard: Dashboard, parentFolder: FolderInfo): Seq[ValidationMessage] = {
    val info = CiInfo(dashboard, parentFolder)
    val tiles = dashboard.getTiles
    dashboard.setTiles(List.empty[Tile].asJava)

    val dashboardWarnings = tiles.asScala.foldLeft(validateCiProperties(dashboard, info)) {
      case (tileWarnings, tile) => {
        // Validate the tiles in the phase as separate CIs
        val tileInfo = CiInfo(tile, parentFolder, Some(info))
        tileWarnings ++ validateTile(tile, tileInfo)
      }
    }

    dashboard.setTiles(tiles)
    dashboardWarnings
  }

  private def validateTile(tile: Tile, info: CiInfo): Seq[ValidationMessage] = {
    validateCiProperties(tile, info)
  }

  private def validateReleaseFolder(releaseTrigger: ReleaseTrigger,
                                    ci: ConfigurationItem,
                                    ciInfo: CiInfo,
                                    validationMessages: ListBuffer[ValidationMessage],
                                    releaseFolderIsOutOfTriggerScope: Boolean): Unit = {
    val propertyReleaseFolder = ci.getType.getDescriptor.getPropertyDescriptors.asScala
      .find(_.getName == "releaseFolder")
      .map(_.getFqn)
      .getOrElse("releaseFolder")

    if (!releaseFolderIsOutOfTriggerScope) {
      validationMessages += new NonVersionedRef(ciInfo, propertyReleaseFolder, null)
    } else {
      val releaseFolderOpt = Option(releaseTrigger.getReleaseFolder)

      val folderIsLocal = releaseFolderOpt.exists{ folder => folder.startsWith(Ids.LOCAL_FOLDER) || folder.startsWith(releaseTrigger.getFolderId) }

      releaseFolderOpt.filterNot(_ => folderIsLocal).foreach { releaseFolder =>

        val refCiInfo: CiInfo = {
          if (releaseFolder.startsWith(Ids.ROOT_FOLDER_ID + Ids.SEPARATOR)) {
            val folderPath = folderService.getPath(releaseFolder).drop(1).mkString(Ids.SEPARATOR)
            createRefFolder(folderPath).orNull
          } else {
            createRefFolder(releaseFolder).orNull
          }
        }

        validationMessages += new NonVersionedRef(ciInfo, propertyReleaseFolder, refCiInfo)
      }
    }
  }

  private def validateTriggerTemplateReference(ci: ConfigurationItem, ciInfo: CiInfo, releaseFolderIsOutOfTriggerScope: Boolean): Seq[ValidationMessage] = {
    ci match {
      case releaseTrigger: ReleaseTrigger =>
        val validationMessages = ListBuffer.empty[ValidationMessage]
        if (config.getExportTriggers && !config.getExportTemplates) {
          val messages = Option(releaseTrigger.getTemplate).flatMap { templateTitle =>
            val templates = releaseService.findTemplatesByTitle(releaseTrigger.getFolderId, templateTitle, 0, 2, 1).asScala.toList
            templates.headOption.map { template =>
              val templateCiInfo = CiInfo(template.getId, "xlrelease.Release", "Release", rootFolder.getAbsolutePath,
                template.findFolderId(), template.getTitle)
              val descriptors = ci.getType.getDescriptor.getPropertyDescriptors.asScala
              val propertyTemplate = descriptors.find(_.getName.equals("template")).map(_.getFqn).getOrElse("template")
              new NonVersionedRef(ciInfo, propertyTemplate, templateCiInfo)
            }
          }
          messages.foreach(validationMessages += _)
        }

        validateReleaseFolder(releaseTrigger, ci, ciInfo, validationMessages, releaseFolderIsOutOfTriggerScope)

        validationMessages.toSeq
      case _ => Seq.empty
    }
  }

  private def getExternalReferencesByDescriptorsByCis(parentCi: ConfigurationItem): List[(ConfigurationItem, List[CiDependency])] = {
    CiHelper.getNestedCis(parentCi).asScala.map { ci =>
      val ciDependencies = ci.getType.getDescriptor.getPropertyDescriptors.asScala.view
        .map(pd => pd -> pd.get(ci))
        .filter { case (pd, value) => !pd.isAsContainment && value != null }
        .flatMap { case (pd, value) =>
          pd.getKind match {
            case PropertyKind.CI =>
              val valueCi = value.asInstanceOf[ConfigurationItem]
              if (!CiHelper.isChildViaOneOfChildProperties(valueCi, ci)) {
                Some(CiDependency(pd, List(valueCi)))
              } else {
                None
              }
            case PropertyKind.SET_OF_CI | PropertyKind.LIST_OF_CI =>
              val valueCiCol = value.asInstanceOf[java.util.Collection[ConfigurationItem]].asScala.toList
              if (valueCiCol.nonEmpty) {
                Some(CiDependency(pd, valueCiCol))
              } else {
                None
              }
            case _ => None
          }
        }
      ci -> ciDependencies.toList
    }.toList
  }
}

class ImportValidator(secrets: Map[String, String],
                      definitions: Seq[Definition],
                      rootFolder: FolderInfo,
                      folderService: FolderService,
                      releaseService: ReleaseService,
                      fromApply: Boolean,
                      config: FolderVersioningSettings)
  extends ExportValidator(definitions, rootFolder, folderService, releaseService, fromApply, config) {

  override protected def validatePasswords(ci: ConfigurationItem, pd: PropertyDescriptor, ciInfo: CiInfo): List[ValidationMessage] = {
    val validationMessages = ListBuffer.empty[ValidationMessage]

    def processSecret(str: String) = {
      getPlaceholder(str) match {
        case Some(placeholder) =>
          secrets.get(placeholder) match {
            case Some(secret) => secret
            case None =>
              logger.debug(s"Property on ${pd.getFqn} of $ciInfo has a missing secret (Missing placeholder name '$placeholder')")
              validationMessages.addOne(new MissingPassword(ciInfo, pd.getFqn, placeholder))
              null
          }
        case None => str
      }
    }

    val value = pd.get(ci)
    if (pd.isPassword && value != null) {
      pd.getKind match {
        case PropertyKind.STRING =>
          val processedSecret = processSecret(value.asInstanceOf[String])
          pd.set(ci, processedSecret)
        case PropertyKind.LIST_OF_STRING =>
          val listval = value.asInstanceOf[java.util.List[String]]
          val processedSecrets = listval.asScala.map(processSecret)
          pd.set(ci, processedSecrets.asJava)
        case PropertyKind.SET_OF_STRING =>
          val setval = value.asInstanceOf[java.util.Set[String]]
          val processedSecrets = setval.asScala.map(processSecret)
          pd.set(ci, processedSecrets.asJava)
        case PropertyKind.MAP_STRING_STRING =>
          val mapval = value.asInstanceOf[java.util.Map[String, String]]
          val processedSecrets = mapval.asScala.view.mapValues(processSecret).toMap
          pd.set(ci, processedSecrets.asJava)
        case _ =>
      }
    }
    validationMessages.toList
  }


  private def getPlaceholder(value: String) = {
    if (value.startsWith(s"$VALUE_TAG ")) {
      val placeholder = value.substring(s"$VALUE_TAG ".length)
      Some(placeholder).filter(StringUtils.hasText)
    } else {
      None
    }
  }
}
