package com.xebialabs.xlrelease.ascode.service.generatestrategy

import com.xebialabs.xlrelease.api.v1.filter.{ApplicationEnvironmentFilterScope, EnvironmentFilters}
import com.xebialabs.xlrelease.ascode.utils.Utils
import com.xebialabs.xlrelease.ascode.utils.Utils.filterFolderOnGeneratePermissions
import com.xebialabs.xlrelease.domain.environments.Environment
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.environments.service.EnvironmentService
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_LIVE_DEPLOYMENTS, EDIT_ENVIRONMENT}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
class EnvironmentAsCodeGenerator @Autowired()(environmentService: EnvironmentService,
                                              permissions: PermissionChecker) extends ApplicationEnvironmentGenerateStrategy[Environment] {
  override val generateOrder: Double = 350

  override def isDefinedAt(config: CiGenerateConfig): Boolean = {
    config.generateConfig.generateEnvironments
  }

  override protected def generateFolder(config: CiGenerateConfig): Map[Option[Folder], List[Environment]] = {
    config.folders
      .filter { folder => filterFolderOnGeneratePermissions(config, folder.getId) || permissions.hasPermission(EDIT_LIVE_DEPLOYMENTS, folder.getId) }
      .map { folder =>
        val scope = new EnvironmentFilters(ApplicationEnvironmentFilterScope.FOLDER)
        scope.setFolderId(folder.getId)
        val environments = environmentService.searchEnvironments(scope).asScala.toList
          .filter { ci =>
            config.ciTitle.forall { n => Utils.stringLike(ci.getTitle, n) }
          }
        Some(folder) -> environments
      }
      .toMap
  }

  override protected def generateGlobal(config: CiGenerateConfig): Map[Option[Folder], List[Environment]] = {
    if (config.isAdminUser || permissions.hasGlobalPermission(EDIT_ENVIRONMENT)) {
      val environments = environmentService.searchEnvironments(new EnvironmentFilters(ApplicationEnvironmentFilterScope.GLOBAL)).asScala.toList
      Map(None -> environments)
    } else {
      Map.empty
    }
  }

}
