package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.model._
import com.xebialabs.ascode.yaml.model.permission.{GlobalPermissionRelation, PermissionRelation, PermissionsSpec}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.PermissionDeniedException
import com.xebialabs.deployit.security.permission.PermissionHelper
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.ascode.Version.XLR_API_VERSION
import com.xebialabs.xlrelease.ascode.metadata.MetadataFields
import com.xebialabs.xlrelease.ascode.service.GenerateService.{EMPTY_DEFINITION_ERROR, GeneratorConfig, generateDefinition}
import com.xebialabs.xlrelease.ascode.service.generatestrategy._
import com.xebialabs.xlrelease.ascode.utils.StaticVariables._
import com.xebialabs.xlrelease.ascode.utils.Utils.processCisForYaml
import com.xebialabs.xlrelease.ascode.yaml.model._
import com.xebialabs.xlrelease.ascode.yaml.model.permission.TaskAccessRelation
import com.xebialabs.xlrelease.ascode.yaml.model.user.XLRUser
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.Ids.{SEPARATOR, normalizeId}
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util.{List => JavaList}
import scala.annotation.tailrec
import scala.collection.mutable
import scala.jdk.CollectionConverters._

sealed trait SearchScope

case class ExactSearch(ids: List[String]) extends SearchScope

case object GlobalSearch extends SearchScope

object FolderSearch {
  def apply(path: String = null, folderId: String = null, includeSubFolders: Boolean = true): FolderSearch = {
    val normalisedPath = if (path == null) null else normalizeId(if (path.endsWith(SEPARATOR)) path.substring(0, path.length - 1) else path)
    new FolderSearch(normalisedPath, folderId, includeSubFolders)
  }
}

case class FolderSearch(path: String = null, folderId: String = null, includeSubFolders: Boolean = true) extends SearchScope

case object AllSearch extends SearchScope

object GenerateService {
  val EMPTY_DEFINITION_ERROR = "Could not generate a definition, there are no results that matches the current input."

  case class GeneratorConfig(
                              name: Option[String],
                              searchScope: SearchScope,
                              cisConfig: CisConfig,
                              isAdmin: Boolean,
                              excludedEntities: Seq[String] = Seq.empty,
                              folderGeneratePermissions: mutable.Map[String, Boolean] = mutable.Map.empty,
                              validateDuplicates: Boolean = false
                            )

  case class CisConfig(
                        generatePermissions: Boolean = false,
                        generateRoles: Boolean = false,
                        generateUsers: Boolean = false,
                        generateEnvironments: Boolean = false,
                        generateApplications: Boolean = false,
                        includeSecrets: Boolean = false,
                        generateTemplates: Boolean = false,
                        generateWorkflows: Boolean = false,
                        generateTriggers: Boolean = false,
                        generateDeliveryPatterns: Boolean = false,
                        generateDashboards: Boolean = false,
                        generateConfigurations: Boolean = false,
                        generateSettings: Boolean = false,
                        generateRiskProfiles: Boolean = false,
                        generateVariables: Boolean = false,
                        generateNotificationSettings: Boolean = false,
                        generateCalendar: Boolean = false,
                        generateFolderVersioning: Boolean = false
                      ) {

    def isEmpty(): Boolean = {
      !hasGlobalTypes() && !hasFolderTypes()
    }

    def hasGlobalTypes(): Boolean = {
      generatePermissions ||
        generateRoles ||
        generateUsers ||
        generateEnvironments ||
        generateApplications ||
        generateTemplates ||
        generateWorkflows ||
        generateDashboards ||
        generateConfigurations ||
        generateSettings ||
        generateRiskProfiles ||
        generateVariables ||
        generateNotificationSettings ||
        generateCalendar
    }

    def hasFolderTypes(): Boolean = {
      generatePermissions ||
        generateTemplates ||
        generateWorkflows ||
        generateTriggers ||
        generateDeliveryPatterns ||
        generateDashboards ||
        generateConfigurations ||
        generateVariables ||
        generateNotificationSettings ||
        generateFolderVersioning ||
        generateApplications ||
        generateEnvironments
    }
  }

  def generateImportsSpecDefinition(imports: List[String]): Definition = {
    if (imports.nonEmpty) {
      generateDefinition(XLR_IMPORT_KIND, ImportsSpec(imports.distinct.sorted), None).get
    } else {
      generateDefinition(XLR_IMPORT_KIND, ImportsSpec(List()), None).get
    }
  }

  private def generateDefinition(kind: String, spec: Spec, home: Option[String]): Option[Definition] = {
    val metadata = home.map(path => Map(MetadataFields.HOMEFOLDER.toString -> path))
    Some(Definition(XLR_API_VERSION, metadata, kind, spec))
  }

}

@Service
class GenerateService(ciGenerator: CiGenerator,
                     permissionsAsCodeService: PermissionsAsCodeService,
                     rolesAsCodeService: RolesAsCodeService,
                     usersAsCodeService: UsersAsCodeService,
                     folderAsCodeService: FolderAsCodeService,
                     generateStrategies: JavaList[GenerateStrategy[_]]) extends Logging {

  private def generatePermissionsDefinition(globalPermissions: List[GlobalPermissionRelation],
                                            taskAccessPermissions: List[TaskAccessRelation],
                                            folderPermissions: List[PermissionRelation],
                                            home: Option[String]): Option[Definition] = {
    if (globalPermissions.nonEmpty || taskAccessPermissions.nonEmpty || folderPermissions.nonEmpty) {
      val permissions = globalPermissions ++ taskAccessPermissions ++ folderPermissions
      generateDefinition(XLR_PERMISSIONS_KIND, PermissionsSpec(permissions), home)
    } else {
      None
    }
  }

  case class HomeWithCis(home: Option[String], cis: List[ConfigurationItem])

  @tailrec
  private def getHomeWithCis(path: Option[String], folder: Folder): HomeWithCis = {
    val children: List[ConfigurationItem] = folder.getChildren.asScala.toList
    children.find(_.getType.instanceOf(Type.valueOf(classOf[Folder]))) match {
      case Some(f) =>
        val childFolder = f.asInstanceOf[Folder]
        getHomeWithCis(path match {
          case Some(p) => Some(s"$p/${childFolder.getTitle}")
          case None => Some(childFolder.getTitle)
        }, childFolder)
      case None => HomeWithCis(path, children)
    }
  }

  private def generateTemplateDefinition(cis: List[ConfigurationItem], references: List[CiReference], home: Option[String]): Option[Definition] = {
    if (cis.nonEmpty) {
      processCisForYaml(cis)
      generateDefinition(XLR_TEMPLATE_KIND, CiSpec(cis, references), home)
    } else {
      None
    }
  }

  private def generateRolesDefinition(roles: List[Role], home: Option[String]): Option[Definition] = {
    if (roles.nonEmpty) {
      generateDefinition(XLR_ROLES_KIND, RolesSpec(roles), home)
    } else {
      None
    }
  }

  private def generateUsersDefinition(users: List[XLRUser], home: Option[String]): Option[Definition] = {
    if (users.nonEmpty) {
      generateDefinition(XLR_USERS_KIND, UsersSpec(users), home)
    } else {
      None
    }
  }

  private def generateEnvironmentsDefinition(generatorConfig: GeneratorConfig, excludeStrategies: List[GenerateStrategy[_]]) = {
    if (generatorConfig.cisConfig.generateEnvironments) {
      val (envCis, envReferences) = ciGenerator.generateCis(generatorConfig, excludeStrategies)
      val envsHomeWithCis: HomeWithCis = getHomeWithCis(generatorConfig, envCis)
      if (envsHomeWithCis.cis.nonEmpty) {
        generateDefinition(XLR_ENVIRONMENTS_KIND, CiSpec(envsHomeWithCis.cis, envReferences), envsHomeWithCis.home)
      } else {
        None
      }
    } else {
      None
    }
  }

  private def generateApplicationsDefinition(generatorConfig: GeneratorConfig, excludeStrategies: List[GenerateStrategy[_]]) = {
    if (generatorConfig.cisConfig.generateApplications) {
      val (appCis, appReferences) = ciGenerator.generateCis(generatorConfig, excludeStrategies)
      val appsHomeWithCis: HomeWithCis = getHomeWithCis(generatorConfig, appCis)
      if (appsHomeWithCis.cis.nonEmpty) {
        generateDefinition(XLR_APPLICATIONS_KIND, CiSpec(appsHomeWithCis.cis, appReferences), appsHomeWithCis.home)
      } else {
        None
      }
    } else {
      None
    }
  }

  def generate(generatorConfig: GeneratorConfig): List[Definition] = {
    if (isGenerateEmpty(generatorConfig)) {
      throw new AsCodeException(s"Please specify a kind or path to generate.")
    }

    if (generatorConfig.cisConfig.includeSecrets && !PermissionHelper.isCurrentUserAdmin) {
      throw PermissionDeniedException.forPermission(ADMIN, null: String)
    }

    val startTimeMillis = System.currentTimeMillis()
    val generatedDefinitions = try {
      val (appEnvStrategies, ciStrategies) = generateStrategies.asScala.toList.partition(_.isInstanceOf[ApplicationEnvironmentGenerateStrategy[_]])
      val (applicationStrategies, environmentStrategies) = appEnvStrategies.partition(_.isInstanceOf[ApplicationAsCodeGenerator])
      val (cis, references) = ciGenerator.generateCis(generatorConfig, excludeStrategies = appEnvStrategies)
      val homeWithCis: HomeWithCis = getHomeWithCis(generatorConfig, cis)

      val templatesDefinition = generateTemplateDefinition(homeWithCis.cis, references, homeWithCis.home)
      val nestedFolders = cis.filter(_.isInstanceOf[Folder]).asInstanceOf[List[Folder]]
      val (globalPermissions, taskAccessPermissions, folderPermissions) = permissionsAsCodeService.generatePermissions(nestedFolders, generatorConfig)
      val permissionsDefinition = generatePermissionsDefinition(globalPermissions, taskAccessPermissions, folderPermissions, homeWithCis.home)
      val roles = rolesAsCodeService.generateGlobalRoles(generatorConfig)
      val rolesDefinition = generateRolesDefinition(roles, homeWithCis.home)
      val users = usersAsCodeService.generateUsers(generatorConfig)
      val usersDefinition = generateUsersDefinition(users, homeWithCis.home)
      val environmentStagesDefinition = generateEnvironmentsDefinition(generatorConfig,
        excludeStrategies = ciStrategies ++ applicationStrategies ++ environmentStrategies.filterNot(_.isInstanceOf[EnvironmentStageAsCodeGenerator]))
      val environmentLabelsDefinition = generateEnvironmentsDefinition(generatorConfig,
        excludeStrategies = ciStrategies ++ applicationStrategies ++ environmentStrategies.filterNot(_.isInstanceOf[EnvironmentLabelAsCodeGenerator]))
      val environmentsDefinition = generateEnvironmentsDefinition(generatorConfig,
        excludeStrategies = ciStrategies ++ applicationStrategies ++ environmentStrategies.filterNot(_.isInstanceOf[EnvironmentAsCodeGenerator]))
      val environmentReservationsDefinition = generateEnvironmentsDefinition(generatorConfig,
        excludeStrategies = ciStrategies ++ applicationStrategies ++ environmentStrategies.filterNot(_.isInstanceOf[EnvironmentReservationAsCodeGenerator]))
      val applicationsDefinition = generateApplicationsDefinition(generatorConfig, excludeStrategies = ciStrategies ++ environmentStrategies)

      List(
        usersDefinition,
        rolesDefinition,
        environmentStagesDefinition,
        environmentLabelsDefinition,
        environmentsDefinition,
        applicationsDefinition,
        environmentReservationsDefinition,
        templatesDefinition,
        permissionsDefinition
      ).flatten
    } finally {
      logger.debug(s"As-code generate took ${System.currentTimeMillis() - startTimeMillis} ms")
    }

    if (generatedDefinitions.isEmpty) {
      throw new AsCodeException(EMPTY_DEFINITION_ERROR)
    }

    generatedDefinitions
  }
  private def getHomeWithCis(generatorConfig: GeneratorConfig, cis: List[ConfigurationItem]): HomeWithCis = {
    val homeWithCis: HomeWithCis = generatorConfig.searchScope match {
      case FolderSearch(path, folderId, _) => {
        val homeCis = cis.find(_.isInstanceOf[Folder])
          .map(_.asInstanceOf[Folder].
            getChildren.asScala.toList)
          .getOrElse(cis)
        if (path != null) {
          HomeWithCis(Some(path), homeCis)
        } else {
          HomeWithCis(Some(folderAsCodeService.getFolderPath(folderId)), homeCis)
        }
      }
      case ExactSearch(_) => {
        cis.find(_.isInstanceOf[Folder]).map(_.asInstanceOf[Folder]) match {
          case Some(folder) => getHomeWithCis(Some(folder.getTitle), folder)
          case None => HomeWithCis(None, cis)
        }
      }
      case _ => HomeWithCis(None, cis)
    }
    homeWithCis
  }

  // scalastyle:off cyclomatic.complexity
  private def isGenerateEmpty(generatorConfig: GeneratorConfig): Boolean = {
    generatorConfig.searchScope == AllSearch &&
      generatorConfig.name.isEmpty &&
      generatorConfig.cisConfig.isEmpty()
  }

  // scalastyle:on cyclomatic.complexity
}
