package com.xebialabs.xlrelease.ascode.service

import com.xebialabs.ascode.exception.AsCodeException
import com.xebialabs.ascode.yaml.dto.AsCodeResponse.EntityKinds._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.EnvironmentApi
import com.xebialabs.xlrelease.ascode.utils.ImportContext
import com.xebialabs.xlrelease.domain.environments.Environment
import com.xebialabs.xlrelease.domain.versioning.ascode.validation.ValidationMessage
import com.xebialabs.xlrelease.environments.repository.{EnvironmentLabelRepository, EnvironmentRepository, EnvironmentStageRepository}
import com.xebialabs.xltype.serialization.CiReference
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class EnvironmentsAsCodeService(environmentApi: EnvironmentApi,
                                environmentRepository: EnvironmentRepository,
                                environmentLabelRepository: EnvironmentLabelRepository,
                                environmentStageRepository: EnvironmentStageRepository) {

  def process(context: ImportContext, environment: Environment): ImportResult = {
    context.scope.getFolderId match {
      case Some(folderId) if environment.getCorrelationUid != null && !environment.getCorrelationUid.isBlank =>
        environmentRepository.findInFolderByCorrelationId(folderId, environment.getCorrelationUid) match {
          case Some(existing) =>
            update(context, environment, existing)
          case None =>
            create(context, environment)
        }
      case Some(folderId) =>
        val existing = environmentRepository.fetchEnvironmentsByFolderId(folderId).filter(_.getTitle == environment.getTitle)
        if (existing.size > 1) {
          throw new AsCodeException(s"Multiple environments found with the same title [${environment.getTitle}]" +
            s" in folder [$folderId], but no correlationUid is provided")
        } else if (existing.size == 1) {
          update(context, environment, existing.head)
        } else {
          create(context, environment)
        }
      case _ =>
        environmentRepository.findEnvironmentByTitleAndFolder(environment.getTitle, None) match {
          case Some(existing) =>
            update(context, environment, existing)
          case None =>
            create(context, environment)
        }
    }
  }

  private def create(context: ImportContext, environment: Environment) = {
    environment.setFolderId(context.scope.getFolderId.orNull)
    resolveReferences(context.references)
    val messages = validate(context, environment)
    val created = environmentApi.create(environment)
    ImportResult(List(CI.ids.withCreated(created.getId)),
      Seq.empty,
      Map.empty,
      messages
    )
  }

  private def update(context: ImportContext, environment: Environment, existing: Environment) = {
    environment.setId(existing.getId)
    environment.setFolderId(context.scope.getFolderId.orNull)
    if (environment.getCorrelationUid == null && existing.getCorrelationUid != null && !existing.getCorrelationUid.isBlank) {
      environment.setCorrelationUid(existing.getCorrelationUid)
    }
    resolveReferences(context.references)
    if (existing.getDeploymentTarget != null) {
      Option(environment.getDeploymentTarget).foreach(_.setId(existing.getDeploymentTarget.getId))
    }
    val messages = validate(context, environment)
    val updated = environmentApi.update(environment)
    ImportResult(List(CI.ids.withUpdated(updated.getId)),
      Seq.empty,
      Map.empty,
      messages
    )
  }

  private def validate(context: ImportContext, environment: Environment): List[ValidationMessage] = {
    context.validator match {
      case Some(validator) => validator.validateCi(environment, context.getFolderInfo()).toList
      case None => List.empty
    }
  }

  private def resolveReferences(references: List[CiReference]): Unit = {
    references.foreach(reference => {
      reference.getProperty.getName match {
        case "labels" =>
          val labels = reference.getIds.asScala.map(ref => {
            try {
              environmentLabelRepository.findByTitle(ref)
            } catch {
              case _: NotFoundException =>
                environmentLabelRepository.findById(ref)
            }
          }).toList.asJava
          reference.getCi.setProperty("labels", labels)
        case "stage" =>
          val stage = reference.getIds.asScala.map(ref => {
            try {
              environmentStageRepository.findByTitle(ref)
            } catch {
              case _: NotFoundException =>
                environmentStageRepository.findById(ref)
            }
          }).toList.head
          reference.getCi.setProperty("stage", stage)
      }
    })
  }

}
