import React from 'react';
import { DotButton, DotCard, DotCardContent, DotCardFooter, DotCardHeader, DotIconButton, DotTypography } from '@digital-ai/dot-components';
import { AnalyticsCard, AnalyticsCardProps, NOT_CONFIGURED_TOOLTIP } from './analytics-card.component';
import { mockResizeObserver, mountWithTheme, ReactWrapper } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { ContextMenu } from '../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/overview/components/context-menu/context-menu.component';
import { MenuItem } from '../../../../../../../../../core/xlr-ui/app/react/components/menu-item/menu-item.component';
import { CardContextMenuItemKey } from '../types';
import { ChipGroup } from '../../../../../../../../../core/xlr-ui/app/react/components/chip-group/chip-group.component';

describe('AnalyticsCard', () => {
    const id = '655285cdd75b4c64fc5bead2';
    const externalId = 123;
    const embeddingId = '8dd03d24-008c-450c-9bbe-f7a21e2a5033';
    const embeddingUrl = 'https';

    const contentText = 'My content text';
    const imageSrc = 'image src';
    const isFavorite = false;
    const onContextMenuItemClick = jest.fn();
    const onFavoriteClick = jest.fn();
    const onOpenDashboardClick = jest.fn();
    const subtitle = 'My subtitle';
    const tags = ['tag1', 'tag2'];
    const title = 'My card title';

    let wrapper: ReactWrapper;

    const defaultProps: AnalyticsCardProps = {
        embeddingId,
        embeddingUrl,
        externalId,
        hasFolderFilter: false,
        contentText,
        id,
        imageSrc,
        isAnalyticsConfigured: true,
        isFavorite,
        onContextMenuItemClick,
        onFavoriteClick,
        onOpenDashboardClick,
        subtitle,
        tags,
        title,
    };

    const mount = (props: AnalyticsCardProps = defaultProps) => {
        wrapper = mountWithTheme(<AnalyticsCard {...props} />);
    };

    beforeAll(() => {
        mockResizeObserver();
    });

    afterAll(() => {
        jest.restoreAllMocks();
    });

    afterEach(() => {
        jest.resetAllMocks();
    });

    const getDotCard = () => wrapper.find(DotCard);
    const getDotCardHeader = () => getDotCard().find(DotCardHeader);
    const getDotCardContent = () => getDotCard().find(DotCardContent);
    const getFavoriteButton = () => getDotCardHeader().findWhere((node) => node.is(DotIconButton) && node.props()['data-testid'] === 'favorite-button');
    const getContextMenu = () => getDotCardHeader().find(ContextMenu);
    const getImage = () => getDotCard().find('img');
    const getContentTypography = () => getDotCardContent().findWhere((node) => node.is(DotTypography) && node.props()['data-testid'] === 'content-typography');
    const getChipGroup = () => getDotCardContent().find(ChipGroup);
    const getDotCardFooter = () => getDotCard().find(DotCardFooter);
    const getOpenDashboardButton = () => getDotCardFooter().findWhere((node) => node.is(DotButton) && node.props().className === 'open-dashboard-btn');

    describe('default render', () => {
        it('should render with the default props', () => {
            mount();
            expect(getDotCard()).toExist();

            const dotCardHeader = getDotCardHeader();
            expect(dotCardHeader).toExist();
            expect(dotCardHeader.props().className).toBe('card-header');
            expect(dotCardHeader.props().subheader).toBe(`by ${subtitle}`);
            expect(dotCardHeader.props().title).toBe(title);
            expect(dotCardHeader.props().titleSize).toBe('medium');

            const dotCardContent = getDotCardContent();
            expect(dotCardContent).toExist();
            expect(dotCardContent.props().className).toBe('card-content');

            const favoriteButton = getFavoriteButton();
            expect(favoriteButton).toExist();
            expect(favoriteButton.props().iconId).toBe('star-favorites-default');
            expect(favoriteButton.props().tooltip).toBe('Add to favourites');

            const contextMenu = getContextMenu();
            expect(contextMenu.props().menuItems).toStrictEqual([
                {
                    children: <MenuItem endIconId="open-new-tab" iconId="edit" label="Edit" />,
                    key: CardContextMenuItemKey.Edit,
                },
                {
                    children: <MenuItem iconId="share" label="Share" />,
                    key: CardContextMenuItemKey.Share,
                },
                {
                    children: <MenuItem iconId="download" label="Export" />,
                    key: CardContextMenuItemKey.Export,
                },
            ]);

            const image = getImage();
            expect(image).toExist();
            expect(image.prop('alt')).toBe('Analytics card image');
            expect(image.prop('src')).toBe(imageSrc);

            const contentTypography = getContentTypography();
            expect(contentTypography).toExist();
            expect(contentTypography.props().variant).toBe('body1');
            expect(contentTypography.props().children).toBe(contentText);

            const chipGroup = getChipGroup();
            expect(chipGroup).toExist();
            expect(chipGroup.props().labels).toBe(tags);

            const cardFooter = getDotCardFooter();
            expect(cardFooter).toExist();

            const openDashboardButton = getOpenDashboardButton();
            expect(openDashboardButton).toExist();
            expect(openDashboardButton.props().type).toBe('outlined');
        });

        it('should use default subtitle and image if not provided', () => {
            mount({ ...defaultProps, subtitle: '', imageSrc: '' });
            expect(getDotCardHeader().props().subheader).toBe('by Digital.ai');
            expect(getImage().prop('src')).toBe('static/@project.version@/include/analytics-dashboard/assets/custom.svg');
        });

        it('should change favorite icon and tooltip when favorite is set to true', () => {
            mount({ ...defaultProps, isFavorite: true });
            expect(getFavoriteButton().props().iconId).toBe('star-favorites-active');
            expect(getFavoriteButton().props().tooltip).toBe('Remove from favourites');
        });

        it('should trigger correct handler when context item is selected', () => {
            mount();
            const contextMenu = getContextMenu();
            contextMenu.invoke('onSelect')('edit');
            expect(onContextMenuItemClick).toHaveBeenCalledTimes(1);
            expect(onContextMenuItemClick).toHaveBeenCalledWith('edit');
        });

        it('should trigger correct handler when Open dashboard button is clicked', () => {
            mount();
            const openDashboardBtn = getOpenDashboardButton();
            openDashboardBtn.invoke('onClick')();
            expect(onOpenDashboardClick).toHaveBeenCalledTimes(1);
        });
    });

    describe('when analytics is not configured', () => {
        beforeEach(() => {
            mount({ ...defaultProps, isAnalyticsConfigured: false });
        });

        it('should disable favourite button', () => {
            const favButton = getFavoriteButton();
            expect(favButton.props().tooltip).toBe(NOT_CONFIGURED_TOOLTIP);
            expect(favButton.props().disabled).toBe(true);
        });

        it('should disable context menu', () => {
            const contextMenu = getContextMenu();
            expect(contextMenu.props().tooltip).toBe(NOT_CONFIGURED_TOOLTIP);
            expect(contextMenu.props().disabled).toBe(true);
        });

        it('should add data-testid to open dashboard button', () => {
            const openDashboardButton = getOpenDashboardButton();
            expect(openDashboardButton.props()['data-testid']).toBe('open-analytics-dashboard-btn');
        });
    });
});
