import {
    calculateNumberOfFiltersApplied,
    calculateTitleAndSubtitleForEmptyState,
    collectAllAuthors,
    convertArrayToString,
    filterCardsBySearch,
    isFavouriteCard,
    getFavouriteIcon,
    getFavouriteText,
    mapSupersetToAnalyticsCard,
    removeAuthorFromFilters,
    removeCategoryFromFilters,
    hasFolderFilter,
} from './index';
import { CardSearch } from '../ducks/analytics.reducer';
import { PlatformListDashboardResponse } from '../types';

describe('analytics helpers', () => {
    describe('calculateNumberOfFiltersApplied', () => {
        it('should return 0 when cardSearch is undefined', () => {
            const result = calculateNumberOfFiltersApplied(undefined);
            expect(result).toBe(0);
        });

        it('should return 0 when both categories and authorNames are empty arrays', () => {
            const cardSearch: CardSearch = {
                categories: [],
                authorNames: [],
            };
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(0);
        });

        it('should return the total number of filters applied when categories and authorNames have elements', () => {
            const cardSearch: CardSearch = {
                categories: ['Category 1'],
                authorNames: ['Author 1', 'Author 2'],
            };
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(3);
        });

        it('should return the number of filters applied based on the provided categories and authorNames', () => {
            const cardSearch: CardSearch = {
                categories: ['Category 1', 'Category 2'],
                authorNames: ['Author 1'],
            };
            const result = calculateNumberOfFiltersApplied(cardSearch);
            expect(result).toBe(3);
        });
    });

    describe('removeCategoryFromFilters', () => {
        it('should return undefined when filteredCategories is undefined', () => {
            const result = removeCategoryFromFilters('Category 1', undefined);
            expect(result).toBeUndefined();
        });

        it('should return an empty array when filteredCategories is an empty array', () => {
            const result = removeCategoryFromFilters('Category 1', []);
            expect(result).toEqual([]);
        });

        it('should remove the specified category from the filteredCategories', () => {
            const category1 = 'Category 1';
            const category2 = 'Category 2';
            const category3 = 'Category 3';

            const filteredCategories: string[] = [category1, category2, category3];
            const result = removeCategoryFromFilters('Category 2', filteredCategories);
            expect(result).toEqual([category1, category3]);
        });

        it('should not modify the original filteredCategories array', () => {
            const category1 = 'Category 1';
            const category2 = 'Category 2';
            const category3 = 'Category 3';

            const filteredCategories: string[] = [category1, category2, category3];
            removeCategoryFromFilters('Category 2', filteredCategories);
            expect(filteredCategories).toEqual([category1, category2, category3]);
        });
    });

    describe('removeAuthorFromFilters', () => {
        it('should return undefined when filteredAuthorNames is undefined', () => {
            const result = removeAuthorFromFilters('Author 1', undefined);
            expect(result).toBeUndefined();
        });

        it('should return an empty array when filteredAuthorNames is an empty array', () => {
            const result = removeAuthorFromFilters('Author 1', []);
            expect(result).toEqual([]);
        });

        it('should remove the specified authorName from the filteredAuthorNames', () => {
            const authorNames = ['Author 1', 'Author 2', 'Author 3'];
            const result = removeAuthorFromFilters('Author 2', authorNames);
            expect(result).toEqual(['Author 1', 'Author 3']);
        });

        it('should not modify the original filteredAuthorNames array', () => {
            const authorNames = ['Author 1', 'Author 2', 'Author 3'];
            removeAuthorFromFilters('Author 2', authorNames);
            expect(authorNames).toEqual(['Author 1', 'Author 2', 'Author 3']);
        });
    });

    describe('convertArrayToString', () => {
        it('should return undefined if undefined', () => {
            expect(convertArrayToString(undefined)).toBeUndefined();
        });
        it('should return undefined if empty', () => {
            expect(convertArrayToString([])).toBeUndefined();
        });
        it('should return csv if there is something', () => {
            expect(convertArrayToString(['a', 'b'])).toBe('a,b');
        });
    });

    describe('isFavouriteCard', () => {
        it('should return true if card id is present in the favourite dashboards array', () => {
            expect(isFavouriteCard(['1', '2'], '1')).toBe(true);
        });
        it('should return false if card id is not present in the favourite dashboards array', () => {
            expect(isFavouriteCard(['1', '2'], '3')).toBe(false);
        });
        it('should return false if favourite dashboards array is empty', () => {
            expect(isFavouriteCard([], '1')).toBe(false);
        });
    });

    describe('calculateTitleAndSubtitleForEmptyState', () => {
        it('should return correct title and subtitle when on favourite tab', () => {
            const favouriteTabData = {
                subtitle: 'To easily add dashboards to your favourites, simply click the star icon located next to each of the dashboards. 🌟',
                title: 'No favourites have been added so far',
            };
            expect(calculateTitleAndSubtitleForEmptyState(true, true)).toStrictEqual(favouriteTabData);
            expect(calculateTitleAndSubtitleForEmptyState(true, false)).toStrictEqual(favouriteTabData);
        });
        it('should return correct title and subtitle when not on favourite tab, depending if analytics connection is configured', () => {
            expect(calculateTitleAndSubtitleForEmptyState(false, true)).toStrictEqual({
                subtitle: 'Create and manage Digital.ai Analytics dashboards',
                title: 'Digital.ai Analytics dashboards',
            });
            expect(calculateTitleAndSubtitleForEmptyState(false, false)).toStrictEqual({
                subtitle: 'There are no available dashboards yet.',
                title: 'No dashboards yet',
            });
        });
    });

    describe('filterCardsBySearch', () => {
        const cards = [
            {
                title: 'My card',
                subtitle: 'digital',
                tags: ['devops'],
            },
            {
                title: 'My other card',
                subtitle: 'custom',
                tags: ['devops'],
            },
        ] as never;
        it('should return the same if no filter', () => {
            expect(filterCardsBySearch(cards, undefined)).toStrictEqual(cards);
        });
        it('should filter by name', () => {
            expect(filterCardsBySearch(cards, { name: 'My other card' })).toStrictEqual([
                {
                    title: 'My other card',
                    subtitle: 'custom',
                    tags: ['devops'],
                },
            ]);
        });
        it('should filter by authorNames', () => {
            expect(filterCardsBySearch(cards, { authorNames: ['digital'] })).toStrictEqual([{ subtitle: 'digital', tags: ['devops'], title: 'My card' }]);
        });
        it('should filter by categories', () => {
            expect(filterCardsBySearch(cards, { categories: ['devops'] })).toStrictEqual(cards);
        });
    });

    describe('hasFolderFilter', () => {
        const mockDashboard: PlatformListDashboardResponse = {
            created_dt: '',
            dashboard_url: '',
            description: '',
            external_embedding_id: '',
            external_id: '',
            filter_configuration: [
                {
                    name: '',
                    id: '',
                    targets: [{ column: { name: 'test' } }, {}],
                },
                {
                    name: '',
                    id: '',
                } as never,
            ],
            id: '',
            lifecycle_state: 'DRAFT',
            name: '',
            server_url: '',
        };

        it('should return true if folderColumn is in native filter configuration targets', () => {
            const result = hasFolderFilter(mockDashboard, 'test');
            expect(result).toBe(true);
        });

        it('should return false if folderColumn is not in native filter configuration targets', () => {
            const result = hasFolderFilter(mockDashboard, 'otherColumn');
            expect(result).toBe(false);
        });
    });

    describe('mapSupersetToAnalyticsCard', () => {
        const response = [
            {
                id: '14',
                external_id: '14',
                name: 'name',
                created_dt: 'created_dt',
                description: 'description',
                external_embedding_id: 'external_embedding_id',
                server_url: 'api/v1/dashboard/14',
            },
        ] as Array<PlatformListDashboardResponse>;
        it('should map as much as possible', () => {
            expect(mapSupersetToAnalyticsCard(response, '')).toStrictEqual([
                {
                    contentText: 'description',
                    hasFolderFilter: false,
                    embeddingId: 'external_embedding_id',
                    embeddingUrl: 'api/v1/dashboard/14',
                    externalId: 14,
                    id: '14',
                    imageSrc: 'TODO',
                    subtitle: 'TODO',
                    tags: ['TODO'],
                    title: 'name',
                },
            ]);
        });
    });

    describe.skip('collectAllAuthors', () => {
        const response = [
            {
                changed_by_name: 'author',
            },
            {
                changed_by_name: 'author',
            },
            {
                changed_by_name: 'author 1',
            },
        ] as never;
        it('should return a list with all unique authors', () => {
            expect(collectAllAuthors(response)).toStrictEqual(['author', 'author 1']);
        });
    });

    describe('getFavouriteIcon', () => {
        it('should return icon', () => {
            expect(getFavouriteIcon(true)).toBe('star-favorites-active');
            expect(getFavouriteIcon(false)).toBe('star-favorites-default');
        });
    });

    describe('getFavouriteText', () => {
        it('should return button text', () => {
            expect(getFavouriteText(true)).toBe('Remove from favourites');
            expect(getFavouriteText(false)).toBe('Add to favourites');
        });
    });
});
