import React from 'react';
import { mountWithTheme, ReactWrapper } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { AnalyticsPageFilterDrawer, AnalyticsPageFilterDrawerProps } from './analytics-page-filter-drawer.component';
import { CardSearch } from '../ducks/analytics.reducer';
import { DotButton, DotCheckbox, DotDrawer, DotIconButton, DotTypography, DotAutoComplete, DotChip } from '@digital-ai/dot-components';

describe('AnalyticsPageFilterDrawer', () => {
    let wrapper: ReactWrapper;
    const authors = ['author1', 'author2', 'author3'];
    const categories = ['Cat1', 'Cat2', 'Cat3', 'Cat4'];
    const cardSearch: CardSearch = { categories: [categories[0], categories[1]], authorNames: [authors[0]] };
    const onCardFilter = jest.fn();
    const onClearAll = jest.fn();
    const onDrawerClose = jest.fn();

    const defaultProps: AnalyticsPageFilterDrawerProps = {
        authors,
        cardSearch,
        categories,
        isDrawerOpened: true,
        numberOfFiltersApplied: 3,
        onCardFilter,
        onClearAll,
        onDrawerClose,
    };

    const getDotDrawer = () => wrapper.find(DotDrawer);
    const getAnalyticsDrawerHeaderDiv = () => wrapper.find('div.analytics-drawer-header');
    const getFiltersTitleTypography = () => getAnalyticsDrawerHeaderDiv().find(DotTypography);
    const getCloseDrawerButton = () => getAnalyticsDrawerHeaderDiv().find(DotIconButton);

    const getAnalyticsDrawerActionsDiv = () => wrapper.find('div.analytics-drawer-actions');
    const getAppliedFiltersTitleTypography = () => getAnalyticsDrawerActionsDiv().find(DotTypography);
    const getClearAllButton = () => getAnalyticsDrawerActionsDiv().find(DotButton);

    const getFilterGroupTitleDiv = () => wrapper.find('div.filter-group-title');
    const getFilterByCategoryTitleTypography = () => getFilterGroupTitleDiv().at(0).find(DotTypography);

    const getFilterCheckboxGroup = () => wrapper.find('div.filter-checkbox-group');

    const getCategoriesCheckboxes = () => getFilterCheckboxGroup().at(0).find(DotCheckbox);

    const getAuthorAutocomplete = () => getFilterCheckboxGroup().at(1).find(DotAutoComplete);
    const getAuthorAutocompleteChips = () => getAuthorAutocomplete().find(DotChip);

    const mount = (props = defaultProps) => {
        wrapper = mountWithTheme(<AnalyticsPageFilterDrawer {...props} />);
    };

    afterEach(() => jest.resetAllMocks());

    it('should render with default props', () => {
        mount();
        const drawer = getDotDrawer();
        expect(drawer).toExist();
        const drawerProps = drawer.props();
        expect(drawerProps.ModalProps).toStrictEqual({ hideBackdrop: true });
        expect(drawerProps.PaperProps).toStrictEqual({
            style: {
                height: 'calc(100% - 104px)',
                padding: '0',
                top: '104px',
            },
        });
        expect(drawerProps.anchor).toBe('right');
        expect(drawerProps.className).toBe('analytics-drawer');
        expect(drawerProps.open).toBe(true);
        expect(drawerProps.width).toBe('320px');

        expect(getAnalyticsDrawerHeaderDiv()).toExist();

        const filtersTitleTypography = getFiltersTitleTypography();
        expect(filtersTitleTypography).toExist();
        expect(filtersTitleTypography.props().variant).toBe('h3');
        expect(filtersTitleTypography.props().children).toBe('Filters');

        const closeDrawerButton = getCloseDrawerButton();
        expect(closeDrawerButton).toExist();
        const closeDrawerButtonProps = closeDrawerButton.props();
        expect(closeDrawerButtonProps.iconId).toBe('close');
        expect(closeDrawerButtonProps.tooltip).toBe('Close');
        closeDrawerButton.invoke('onClick')?.({} as never);
        expect(onDrawerClose).toHaveBeenCalledTimes(1);

        expect(getAnalyticsDrawerActionsDiv()).toExist();

        const appliedFiltersTitleTypography = getAppliedFiltersTitleTypography();
        expect(appliedFiltersTitleTypography).toExist();
        expect(appliedFiltersTitleTypography.props().variant).toBe('subtitle2');
        expect(appliedFiltersTitleTypography.props().children).toBe('Applied filters (3)');

        const clearAllButton = getClearAllButton();
        expect(clearAllButton).toExist();
        expect(clearAllButton.props().type).toBe('outlined');
        expect(clearAllButton.props().children).toBe('Clear all');
        clearAllButton.invoke('onClick')?.({} as never);
        expect(onClearAll).toHaveBeenCalledTimes(1);

        expect(getFilterGroupTitleDiv()).toExist();

        const filterByCategoryTitleTypography = getFilterByCategoryTitleTypography();
        expect(filterByCategoryTitleTypography).toExist();
        expect(filterByCategoryTitleTypography.props().variant).toBe('overline');

        expect(getFilterCheckboxGroup()).toExist();
        const categoriesCheckboxes = getCategoriesCheckboxes();
        expect(categoriesCheckboxes.length).toBe(categories.length);

        expect(categoriesCheckboxes.at(0).props().checked).toBe(true);
        expect(categoriesCheckboxes.at(0).props().label).toBe(categories[0]);

        expect(categoriesCheckboxes.at(1).props().checked).toBe(true);
        expect(categoriesCheckboxes.at(1).props().label).toBe(categories[1]);

        expect(categoriesCheckboxes.at(2).props().checked).toBe(false);
        expect(categoriesCheckboxes.at(2).props().label).toBe(categories[2]);

        expect(categoriesCheckboxes.at(3).props().checked).toBe(false);
        expect(categoriesCheckboxes.at(3).props().label).toBe(categories[3]);

        const authorFilterTitle = getFilterCheckboxGroup().at(1);
        expect(authorFilterTitle).toExist();

        const authorAutocomplete = getAuthorAutocomplete().props();

        expect(authorAutocomplete.options).toStrictEqual([{ title: 'author1' }, { title: 'author2' }, { title: 'author3' }]);
        expect(authorAutocomplete.value).toStrictEqual([{ title: 'author1' }]);
        expect(authorAutocomplete.freesolo).toBe(false);
        expect(authorAutocomplete.multiple).toBe(true);

        const authorChips = getAuthorAutocompleteChips();
        expect(authorChips.length).toBe(cardSearch.authorNames?.length);
    });

    it('should render empty DotDrawer when isDrawerOpened is set to false', () => {
        mount({ ...defaultProps, isDrawerOpened: false });
        const drawer = getDotDrawer();
        const drawerProps = drawer.props();
        expect(drawerProps.children).toBe(null);
    });

    it('should execute onCardFilter handler with correct props when category checkbox is checked', () => {
        mount();
        const categoriesCheckboxes = getCategoriesCheckboxes();
        categoriesCheckboxes.at(3).invoke('onChange')?.({ target: { checked: true } } as never, '123');
        expect(onCardFilter).toHaveBeenCalledWith({
            ...cardSearch,
            categories: [categories[0], categories[1], categories[3]],
        });
    });

    it('should execute onCardFilter handler with correct props when category checkbox is unchecked', () => {
        mount();
        const categoriesCheckboxes = getCategoriesCheckboxes();
        categoriesCheckboxes.at(0).invoke('onChange')?.({ target: { checked: false } } as never, '123');
        expect(onCardFilter).toHaveBeenCalledWith({ ...cardSearch, categories: [categories[1]] });
    });

    it('should execute onCardFilter handler with correct props when author is selected', () => {
        mount();
        const authorAutocomplete = getAuthorAutocomplete();
        authorAutocomplete.invoke('onChange')?.('' as never, [{ title: authors[0] }, { title: authors[1] }], '' as never);
        expect(onCardFilter).toHaveBeenCalledWith({ ...cardSearch, authorNames: [authors[0], authors[1]] });
    });

    it('should execute onCardFilter handler with correct props when author is removed from the chip', () => {
        mount();
        const authorChip = getAuthorAutocompleteChips().at(0);
        authorChip.invoke('onDelete')?.('' as never);
        expect(onCardFilter).toHaveBeenCalledWith({ ...cardSearch, authorNames: [] });
    });
});
