import React, { useEffect } from 'react';
import { Provider } from 'react-redux';
import { DotDashboardHeader, DotProgress, DotSnackbarProvider, DotThemeProvider } from '@digital-ai/dot-components';
import { store } from '../../../analytics-store';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { analytics, getAnalyticsState } from '../../ducks/analytics.reducer';
import { Folder } from '../../../../../../../../../../core/xlr-ui/app/types';
import { MicrostrategyDashboard } from './microstrategy/microstrategy-dashboard.component';
import { MicrostrategyProvider } from './microstrategy/microstrategy.provider';
import { AnalyticsAuthProvider } from './analytics-auth-provider.component';
import './analytics-dashboard-page.component.less';

const { loadDashboard, clearSelectedDashboard, favoriteDashboardFromDetails } = analytics.actions;

export interface AnalyticsDashboardPageProps {
    dashboardId: string;
    folder?: Folder;
}

export const AnalyticsDashboardPageProvider = (props: AnalyticsDashboardPageProps) => {
    return (
        <Provider store={store}>
            <DotThemeProvider>
                <DotSnackbarProvider>
                    <AnalyticsAuthProvider>
                        <AnalyticsDashboardPage {...props} />
                    </AnalyticsAuthProvider>
                </DotSnackbarProvider>
            </DotThemeProvider>
        </Provider>
    );
};

export const AnalyticsDashboardPage = ({ dashboardId, folder }: AnalyticsDashboardPageProps) => {
    const { selectedDashboard, tokenValue } = useAppSelector(getAnalyticsState);
    const isFavorite = !!selectedDashboard?.isFavorite;
    const dispatch = useAppDispatch();

    if (folder) {
        // TODO check permissions
    }

    useEffect(() => {
        dispatch(loadDashboard(dashboardId));
        return () => {
            dispatch(clearSelectedDashboard());
        };
    }, []);

    const renderLoader = () => {
        return (
            <div className="dashboard-loader">
                <DotProgress ariaLabel="Loading dashboard data" size={100} />
            </div>
        );
    };

    const handleFavoriteClick = () => {
        dispatch(favoriteDashboardFromDetails({ id: dashboardId, isUnfavorite: isFavorite }));
    };

    const handleBackClick = () => {
        if (folder) {
            window.location.href = `#/folders/${folder.id}/analytics`;
        } else {
            window.location.href = '#/analytics';
        }
    };

    return (
        <div className="analytics-dashboard-page">
            {selectedDashboard?.originalDashboard && tokenValue && (
                <>
                    <DotDashboardHeader
                        canEdit={false}
                        dashboard={selectedDashboard.originalDashboard}
                        isEdit={false}
                        onClose={handleBackClick}
                        onFavorite={handleFavoriteClick}
                    />
                    <MicrostrategyProvider>
                        <MicrostrategyDashboard embedConfig={selectedDashboard.embedConfig} folder={folder} serverUrl={selectedDashboard.serverUrl} />
                    </MicrostrategyProvider>
                </>
            )}
            {!selectedDashboard && renderLoader()}
        </div>
    );
};
